;+
;
; NAME
;  pixtrace2
;
; PURPOSE
;  Maps spectral ratios from ISM calibrated data files in geographic coordinates
;  (Mercator projection). 
;  Allows to plot any spectral parameters (not only ratios) using a hand-written 
;  function.
;
; CALLING SEQUENCE:
;	pixtrace, Fen
;
; INPUT PARAMETERS:
;  Fen: string, nickname of the session.
;
; KEYWORD PARAMETERS:
;  ODD: uses odd channels (default is even channels).
;  ALBEDO: Performs a Minnaert correction to displays albedo when only
;   one series of channels is entered.
;  AIRMASS: Corrects the ratio for airmass variations. This should be used
;   only when displaying an atmospheric band (ie: ratios, with first series 
;   including channels inside a CO2 band).
;  SQUARE: Plots the square of the ratio (useful when computing a band depth 
;   using continuum estimated from a single shoulder). 
;  USER: allows the use of a hand-written function to compute spectral parameter.
;    The function used is called SPECTRAL_PARAM (at the beginning of this file), and must
;    be written along the lines given in comments.
;  SMOOTH: smoothes image with the four nearest neighbours and double weight for
;   the central point. Accounts for image borders only in IDL > 3.6 (limitations in 
;   older versions of convol); otherwise, image size is reduced.
;  NONOISE: plots classes with width of 1 sigma minimum (reduces the number of colors to 
;   the number of significant classes).
;  ANGLE: permits to plot phase, emergence or incidence.
;  SILENT: inhibits extra messages (default).
;  PS: writes a PostScript file with the map.
;
; PROCEDURE:
;	 Two series of channels are entered. The routines computes the geometric 
;   average of intensities in each series, and plots the ratio between these two
;   quantities.
;	 Smoothing is performed with the four neiboring pixels cross-track and 
;   along the track. It is done after byte conversion, and plotted with the same
;   limit values. It may rseult in spurious values wherever a pixel has a zero
;   value (should be filtered out in the convolution).
;
; SUBROUTINES:
;  Make calls to ISM_sess, ISM_lam, ISM_noise, ISM_data, and ISM_coor
;
; LIMITATIONS:
;	1) When plotting on screen, the number of colors used can be smaller than 
;  expected if no plot has been done before. Therefore, the color bar can be dilated
;  and the color scale can be slightly saturated (but values are ok). Just run it again
;  to get a correct plot.
;  2) Longitudes are labeled according to terrestrial convention, due to limitations
;  in map_set.pro. Use pixtrace.pro to label longitudes using planetary convention
;  and overplot Mars contours (requires IAS planetary library).
;
; MODIFICATION HISTORY:
;	Stephane Erard, May 1999
;  SE, may 2001: fixed grid plotting
;-
;*************************************************************************

Function SPECTRAL_PARAM, Tab, sigtot, titre, Param, ind

; Function used to compute spectral parameters other than ratios.
; Should return an array ima with session dimensions, set to 0 
; where no data is available.
; If the parameter is a ratio of two quantities, numerator and denominator 
; should be computed separately, and the ratio should be computed
; only where data is available (pixels pointed by ind) to prevent errors.
; BEWARE that TAB indices used here are channel numbers -1 (ie, to 
; use channel # 10, type TAB.SPEC(9) ).
; Variable Titre can be used to modify plot title.
; Variable sigtot can be used to compute relative variance (ie, (dI/I)^2 ). 

; EXAMPLE: 
; To plot ratio of channels 21 to 23, type
;     ima1=Tab.spec(20)/32768.*.5
;     ima2=Tab.spec(22)/32768.*.5
;     ima=fltarr(param.nsamples, param.nlines)         ; declare correct dimensions
;     ima(ind)=ima1(ind)/ima2(ind)               ;  compute ratio where there is data
;     sigtot = (1./1000.)^2                                           ; relative variance
;     titre=titre+ ' user function'
;     return, ima


ima1=Tab.spec(20)/32768.*.5
ima2=Tab.spec(22)/32768.*.5
ima=fltarr(param.nsamples, param.nlines)         ; declare correct dimensions
ima(ind)=ima1(ind)/ima2(ind)               ;  compute ratio where there is data
sigtot = (1./1000.)^2                                           ; relative variance
titre=titre+ ' user function'
return, ima
end


PRO pixtrace2, Fen, ODD=ODD, PS=ps, SILENT=silent, ALBEDO= albedo, AIRMASS=airmass,$
 ANGLE=angle, SMOOTH=smooth, SQUARE=square, $
 USER=user, NONOISE=nonoise


;t0= systime(1)
si=size(fen)
if si(si(0)+1) ne 7 then begin 
 fen='   '
 print, 'Please give the acronym of the session (without quotes)'
 read, format='(A3)',fen
endif
fen=STRUPCASE(fen)

if n_elements(silent) eq 0 then silent=1
param=ISM_sess(fen, error)     ; read session parameters
IF (error NE 0) THEN RETURN

CoefPS=1.
coultraits=2
if keyword_set(ps) then begin
 devori=!D.name
 set_plot,'ps'
 device,filename='Pixtrace.ps',/color,bit=8,/landscape,/helvetica
 oldfont=!P.font
 !P.font=0
 CoefPS=35.
endif


IF Keyword_Set(odd) THEN BEGIN
 Param.DataFile= Param.ID+'ODD.CAL'
 lam=ISM_lam(error,/ODD)
ENDIF ELSE lam=ISM_lam(error)             ; read wavelength file
IF (error NE 0) THEN RETURN


tab=ISM_data(param, error)     ; read calibrated data
IF (error NE 0) THEN RETURN
coord=ISM_coor(param,error)    ; read coordinates file
IF (error NE 0) THEN RETURN

IF Keyword_Set(odd) THEN BEGIN	  ; read signal to noise (average in gain 2)
 sigton=ISM_noise(error,/ODD)
ENDIF ELSE sigton=ISM_noise(error)
IF (error NE 0) THEN RETURN
sigton=sigton/2.*Param.gain


;      *** Color settings ***
 maxcol=!D.N_Colors - 2
 mincol=3
 loadct, 27, /silent
 tvlct, 0,0,0,2 
 coultrait=2

;      *** Window settings ***
if (!D.WINDOW eq -1) and ((!D.FLAGS and 256) ne 0) then window
xmin=0.02
xmax=0.87
ymin=0.4
posmap=[xmin,ymin,xmax,ymin+(xmax-xmin)/3*!D.x_size/!D.y_size]
col_pal=findgen(maxcol-mincol+1)+mincol
pal=replicate(1.,10)#col_pal
xpos= xmax+.06
hnorm=(maxcol-mincol+1.)/!D.y_size*CoefPS
ypos= ymin+(xmax-xmin)/3*!D.x_size/!D.y_size/2.-hnorm/2.

latll=param.lim(3)+5
mars=1
k = 0.7     ; Minnaert's exponent

noy=[[0.,1.,0.],[1.,2.,1.],[0.,1.,0.]]

if (fen eq 'PHO') then begin
 posmap=[.09,.1,.99,.1+.9/120.*80*!D.x_size/!D.y_size]
 latll=55.
 mars=0
 k = 0.55     ; Minnaert's exponent for Phobos
endif
tit1='Reflectance'
if keyword_set(albedo) then tit1= 'Albedo'


ind = where(coord.mirror ne 0)     ; pixels with data
if N_elements(angle) eq 0 then angle=" "
angle = strlowcase(angle)

Case ANGLE of 
'emergence': begin
  maxi=max(Coord(ind).emergence, min=mini)
  titre=Param.region+'!C'+'Map of emergence angles'
  if silent ne 1 then $
    print, 'Min/max emergence angles (degrees):', [mini, maxi]/100.
   ima=(bytscl(Coord.emergence, min=mini, max=maxi, top = (maxcol-mincol +1)))+mincol < maxcol
  end
'incidence': begin
  maxi=max(Coord(ind).incidence, min=mini)
  titre=Param.region+'!C'+'Map of incidence angles'
  if silent ne 1 then $
    print, 'Min/max incidence angles (degrees):', [mini, maxi]/100.
   ima=(bytscl(Coord.incidence, min=mini, max=maxi, top = (maxcol-mincol +1)))+mincol < maxcol
  end
'phase': begin
  maxi=max(Coord(ind).phase, min=mini)
  titre=Param.region+'!C'+'Map of phase angles'
  if silent ne 1 then $
    print, 'Min/max phase angles (degrees):',[mini, maxi]/100.
  ima=(bytscl(Coord.phase, min=mini, max=maxi, top = (maxcol-mincol +1)))+mincol < maxcol
  end 
else: begin
  par = 'even'
  if keyword_set(odd) then par ='odd'
  
  if keyword_set(USER) then begin
   titre=Param.region+'!C'
   sigtot=0.
   ima=spectral_param(Tab, sigtot, Titre, Param, ind) 
  endif else begin
  
  ch1=1
  j1=0
  numer=(deno=intarr(20))
  Print, 'Enter channel # (1 to 64), 0 when done
  while ch1 ne 0 do begin
   Print, 'Channel #, numerator?'
   read, ch1
   numer(j1)=ch1-1
   j1=j1+1
  endwhile
  ch2=1
  j2=0
  while ch2 ne 0 do begin
   Print, 'Channel #, denominator?'
   read, ch2
   deno(j2)=ch2-1
   j2=j2+1
  endwhile
  j1=j1-1
  j2=j2-1
  print, ' ' 
  print, format='("Map of '+par+' channels ")'
  for j=0,j1-1 do $
  print, format='(" ",I0," (Wavelength = ",F8.4," micron)")',numer(j)+1,lam(numer(j))
  if j2 ne 0 then begin
   print, format='("Divided by channels")'
   for j=0,j2-1 do $
   print, format='(" ",I0," (Wavelength = ",F8.4," micron)")',deno(j)+1,lam(deno(j))
  endif
    
  ima1=Tab.spec(numer(0))/32768.*.5
  for j=1,j1-1 do ima1=Tab.spec(numer(j))/32768.*.5*ima1
  ima1=ima1^(1./float(j1))
  ima=fltarr(param.nsamples, param.nlines)
  sigtot1=1./sigton(numer(0))^2.
  for j=1,j1-1 do sigtot1=1./sigton(numer(j))^2.+sigtot1
  sigtot1=sigtot1/(j1)^2
  if j2 ne 0 then begin
    ima2=Tab.spec(deno(0))/32768.*.5
    for j=1,j2-1 do ima2=Tab.spec(deno(j))/32768.*.5*ima2
    sigtot2=1./sigton(deno(0))^2.
    for j=1,j2-1 do sigtot2=1./sigton(deno(j))^2.+sigtot2
    sigtot = sigtot2/(j2)^2 + sigtot1
    ima2=ima2^(1./float(j2)) 
    ima(ind)=ima1(ind)/ima2(ind)
    if keyword_set(square) then ima=ima^2
  endif else begin
    ima=ima1
    sigtot = sigtot1
    if keyword_set(albedo) then begin
      print, format=$
      '("Geometric albedo estimated with Minnaert''s model (k = ",F4.2,")")',k
       Coord.incidence=coord.incidence < 8500	; for Phobos
       ima = ima*cos(Coord.emergence*(!dtor/100.))^(1.-k)/cos(Coord.incidence*(!dtor/100.))^k
;       ima =cos(Coord.emergence*(!dtor/100.))^(1.-k)/cos(Coord.incidence*(!dtor/100.))^k
    endif
  endelse
  if keyword_set(airmass) then begin
     print, "Airmass correction performed"
     airmass=0.5*(1./cos(Coord.emergence*(!dtor/100.)) + 1./cos(Coord.incidence*(!dtor/100.)))
     ima=1. - (1.-ima)/airmass
;     ima=airmass
  endif

   titre=Param.region+'!C'+par+' channels : '+strcompress(string(numer(0)+1),/remove_all)
   for j=1,j1-1 do titre=titre+'*'+strcompress(string(numer(j)+1),/remove_all)
   if j2 ne 0 then begin
    titre = titre +' / '+strcompress(string(deno(0)+1),/remove_all)
    for j=1,j2-1 do titre=titre+'*'+strcompress(string(deno(j)+1),/remove_all)
   endif else begin
    if keyword_set(albedo) then titre = titre + ' (albedo)'
   endelse
 endelse

   maxi=max(ima(ind), min=mini)
   ecarttype=stdev(ima(ind), moyenne)
   sigtot=sqrt(sigtot)*moyenne
   nbclass=(maxi-mini)/sigtot
   if keyword_set(nonoise) and (nbclass lt (maxcol-mincol +1)) then begin
     ima=long(bytscl(ima, min=mini, max=maxi, top = nbclass)*(maxcol-mincol +1)/nbclass)+mincol < maxcol
     col_pal=byte(findgen(maxcol-mincol+1)/(maxcol-mincol +1)*nbclass)*(maxcol-mincol +1)/nbclass+mincol     ; resample color bar
     pal=replicate(1.,10)#byte(col_pal)
   endif else ima=(bytscl(ima, min=mini, max=maxi, top = (maxcol-mincol +1)))+mincol < maxcol
   if keyword_set(smooth) then begin
    if (!version.release gt 3.6) then begin
     ima=convol(ima, noy,total(noy), /edge_truncate)
    endif else ima=convol(ima, noy,total(noy))
    ind=where(ima ne 0)
    Print, 'Image smoothed'
   endif
   print, ' ' 
   print, 'Min/max value:', [mini, maxi]
   print, 'Mean/std dev:', [moyenne,ecarttype]
   print, 'Noise (assuming all channels are different):', sigtot
   print, 'Total range is',nbclass,' x noise'
  end
endcase


map_set,limit=Param.lim,position=posmap,/mercator,$
 title=titre,  color=coultrait

if silent ne 1 then print, 'Missing pixels:'
lon=(lat=fltarr(4))
nbf=0

for j=0,Param.nLines*Param.nSamples-1 do begin
if (coord(j).mirror gt 0) then begin
 nbf=nbf+1
 for i=0,3 do begin
  lon(i)=coord(j).coord(2*i)/100.
  if (fen eq 'PHO' and lon(i) lt 0.) then lon(i)=180.
  lat(i)=coord(j).coord(2*i+1)/100.
 endfor 
 col = ima(j)
 if (min(lon) ne 0.) then polyfill,lon,lat,color=col  ; security only for Phobos
endif else if silent ne 1 then print, j
endfor
if silent ne 1 then print, 'nb of spectra:', nbf


map_set,limit=Param.lim,position=posmap,/mercator,/noerase
map_grid,/label, lonlab=Param.lim(0)-4,latlab=latll,londel=10.,latdel=10., $
 color=coultrait, clip=0

tv, pal, xpos, ypos, xs=10./!D.x_size*coefPS , ys=hnorm , /normal
xyouts, xpos+0.02, ypos, strtrim(string(mini,format='(F5.3)'),2),/normal,color=coultrait
xyouts,xpos+0.02,ypos+hnorm,strtrim(string(maxi,format='(F5.3)'),2),/normal,color=coultrait
xcadre=[xpos,xpos,xpos+10./!D.x_size*coefPS,xpos+10./!D.x_size*coefPS,xpos]
ycadre=[ypos,ypos+hnorm,ypos+hnorm,ypos,ypos]
plots, xcadre, ycadre, /normal, color=coultrait
xyouts, xpos+0.02, ypos+hnorm*(moyenne-mini)/(maxi-mini), $
 strtrim(string(moyenne,format='(F5.3)'),2), /normal, color=coultrait
plots, [xpos, xpos+10./!D.x_size],[ypos+hnorm*(moyenne-mini)/(maxi-mini),$
 ypos+hnorm*(moyenne-mini)/(maxi-mini)],/normal, color=coultrait

if keyword_set(ps) then begin
 device, /close
 set_plot,devori
 !P.font=oldfont
endif
;print, systime(1)-t0

stop
END
