;	Stephane Erard, july 1995;+

; NAME
; litpav

; PURPOSE
; Reads and maps ISM altimetry data base (Pavonis session only). The measures do not
;  correspond to ISM pixels: the plane was resampled in longitude and 
;  latitude with a step of ~0.3 degrees. Pressure and elevation were
;  computed from the 2.0 micron band depth, using a standard temperature 
;  profile. Accuracies are ~0.01 mb in pressure and ~100 m in elevation.
;	Displays a part of one of the two high resolution image cubes
;	 (Pavonis' caldera).
;	The image is coded using the available number of colors, except color 0
;	Instructions to plot a contour image are commented out.


; CALLING SEQUENCE:
;	litpav, [ima, tab]

; INPUT PARAMETERS:
; none

; OUTPUT PARAMETERS:
; ima:  data in image format
; tab(0:1,*)   : array containing grid information (line and sample; this grid
;	is superimpossed on the images, and differs from that in edited or 
;	calibrated data files)
; tab(2:3,*)   : array containing latitude and longitude of points
; tab(4,*)     : secante (1/cos i + 1/cos e)
; tab(5,*)     : array containing pressure measurements (mb). -5.0 if no measure
; tab(6,*)     : array containing altimetry measurements (km)


;-
;********************************************************************

pro litpav, ima, tab


noms='PAVALT.DAT'

; limc contains the limits of the map: [latmin,longmin,latmax,longmax]
; A ratio of 1/3 between lat and long ranges will preserve the scale
if noms eq 'PAVALT.DAT' then limc=[-.2,-118,.8,-115]
;if noms eq 'BIBALT.DAT' then limc=[-5,-145,5,-115]

if noms eq 'PAVALT.DAT' then numspec=90 ; Total number of samples
;if noms eq 'BIBALT.DAT' then numspec=1267

tab=fltarr(7,numspec)
lat=fltarr(4)
lon=fltarr(4)

IF (!version.release gt 3.6) THEN BEGIN
 CASE !version.os_family OF
  'vms' :dirdata= !ISM_dir+"DRV_DATA.TOPO]"
  'unix' :dirdata=!ISM_dir+"DRV_DATA/TOPO/"
  'MacOS' :dirdata=!ISM_dir+"DRV_DATA:TOPO:"
 ELSE: dirdata=""     ; assumes data in the same directory
 ENDCASE
ENDIF ELSE dirdata=""     ; assumes data in the same directory

openR,  unit, /get_lun, dirdata+noms
readf, unit, format='(2I4,2F8.2,F6.3,F5.2,F5.1)',tab
free_lun, unit


numline=max(tab(0,*))
sperl=max(tab(1,*))
ima=fltarr(sperl,numline)


y=tab(0,*)
x=tab(1,*)
ima(x(*),y(*))=tab(6,*)

tab(3,*)=-tab(3,*)	; terrestrial convention
ind= where(tab(3,*) lt -180, count)
if count ne 0 then tab(3,ind)=360+tab(3,ind)


; Plots a map
loadct, 27
Dx=.9/3
Dy=Dx*!D.x_size/!D.y_size
map_set,limit=limc,position=[.01,.4,0.91,.4+Dy],/mercator,/grid,/label,$
lonlab=limc(0)-.3,latlab=limc(3)+.2,londel=.5,latdel=.2,title=noms


nbcol=!D.table_size-1
pas =0.05
;if noms eq 'BIBALT.DAT' then pas =0.05
if noms eq 'PAVALT.DAT' then pas =0.05
nbf=0
maxi=max(tab(6,*))
mini=min(tab(6,where(tab(5,*) ge 0)))
ran=maxi-mini
col=(tab(6,*)-mini)*nbcol/ran + 1
ind0=where(col ge nbcol)
if (ind0(0) ne -1) then col(ind0)=col(ind0)-1
for j=0,numspec-1 do begin
; if tab(5,j) gt 0 then begin
  nbf=nbf+1
  lon(0)=tab(3,j)+pas
  lon(1)=tab(3,j)-pas
  lon(2)=tab(3,j)-pas
  lon(3)=tab(3,j)+pas
  lat(0)=tab(2,j)+pas
  lat(1)=tab(2,j)+pas
  lat(2)=tab(2,j)-pas
  lat(3)=tab(2,j)-pas
  polyfill,lon,lat,color=col(j)
; endif
endfor
print, 'nb of pixels:', nbf
print, 'Max alti:', maxi, '  Min alti:', mini


; Contour plot of the file
;contour,ima,zrange=[mini,maxi],/follow,nlevels=12

; Diagram
window, 4, xs=640,ys=300
col=reform(col, 15, 6)
lat=reform(tab(2,*),15,6)
lon=reform(tab(3,*),15,6)
shade_surf, col, lon, lat, zrange=[0,800], shades=col,az=200.,ax=30.
surface, col, lon, lat, skirt=0, zrange=[0,800],az=200.,ax=30.,/noerase

end
