;	Stephane Erard, july 1995;+

; NAME
; litalt

; PURPOSE
; Reads and maps ISM altimetry data base. The measures do not
;  correspond to ISM pixels: the plane was resampled in longitude and 
;  latitude with a step of ~0.3 degrees. Pressure and elevation were
;  computed from the 2.0 micron band depth, using a standard temperature 
;  profile. Accuracies are ~0.01 mb in pressure and ~100 m in elevation.
; The image is coded on the available number of colours, except colour 0

; CALLING SEQUENCE:
;	litalt, Fen, [tab]

; INPUT PARAMETERS:
;	Fen: string, nickname of the session.

; OUTPUT PARAMETERS:
; tab(0:1,*)   : array containing grid information (line and sample; this grid
;	is superimpossed on the images, and differs from that in edited or 
;	calibrated data files)
; tab(2:3,*)   : array containing latitude and longitude of points
; tab(4,*)     : secante (1/cos i + 1/cos e)
; tab(5,*)     : array containing pressure measurements (mb). -5.0 if no measure
; tab(6,*)     : array containing altimetry measurements (km)


;-
;********************************************************************

pro litalt, fen, tab



si=size(fen)
if si(si(0)+1) ne 7 then begin 
	fen='   '
	print, 'Please give the acronym of the session (without quotes)'
	read, format='(A3)',fen
endif
noms=STRUPCASE(fen)+'ALT.DAT'


; limc contains the limits of the map: [latmin,longmin,latmax,longmax]
; A ratio of 1/3 between lat and long ranges will preserve the scale
if noms eq 'ARAALT.DAT' then limc=[-10,-40,20,50]
if noms eq 'DAEALT.DAT' then limc=[-30,-180,0,-90]
if noms eq 'SYRALT.DAT' then limc=[-10,40,20,130]
if noms eq 'VMCALT.DAT' then limc=[-20,-140,10,-50]
if noms eq 'AURALT.DAT' then limc=[-20,-100,10,-10]
if noms eq 'OLYALT.DAT' then limc=[0,-180,30,-90]
if noms eq 'GORALT.DAT' then limc=[-10,-180,20,-90]
if noms eq 'ASCALT.DAT' then limc=[-10,-140,20,-50]
if noms eq 'HEBALT.DAT' then limc=[-10,-140,20,-50]

if noms eq 'ARAALT.DAT' then numspec=3146
if noms eq 'DAEALT.DAT' then numspec=3051
if noms eq 'SYRALT.DAT' then numspec=3883
if noms eq 'VMCALT.DAT' then numspec=3808
if noms eq 'AURALT.DAT' then numspec=3248
if noms eq 'OLYALT.DAT' then numspec=2987
if noms eq 'GORALT.DAT' then numspec=3716
if noms eq 'ASCALT.DAT' then numspec=3883
if noms eq 'HEBALT.DAT' then numspec=3937
IF (numspec eq 0) THEN BEGIN
 Message, /CONTINUE, 'Unknown session ID: ' + fen
 return
ENDIF

tab=fltarr(7,numspec)
lat=fltarr(4)
lon=fltarr(4)

IF (!version.release gt 3.6) THEN BEGIN
 CASE !version.os_family OF
  'vms' :dirdata= !ISM_dir+"DRV_DATA.TOPO]"
  'unix' :dirdata=!ISM_dir+"DRV_DATA/TOPO/"
  'MacOS' :dirdata=!ISM_dir+"DRV_DATA:TOPO:"
 ELSE: dirdata=""     ; assumes data in the same directory
 ENDCASE
ENDIF ELSE dirdata=""     ; assumes data in the same directory

openR,  unit, /get_lun, dirdata+noms
readf, unit, format='(2I4,2F8.2,F6.3,F5.2,F5.1)',tab
free_lun, unit



numline=max(tab(0,*))
sperl=max(tab(1,*))
tab(3,*)=-tab(3,*)	; Longitude according to terrestrial convention
ind= where(tab(3,*) lt -180, count)
if count ne 0 then tab(3,ind)=360+tab(3,ind)

;set_plot,'ps'
;device,filename='map.ps',/color,bits=8,/landscape,scale_factor=1.0
;erase


; Plots a map
loadct, 39
Dx=.9/3
Dy=Dx*!D.x_size/!D.y_size
map_set,limit=limc,position=[.01,.4,0.91,.4+Dy],/mercator,/grid,/label,$
lonlab=limc(0)-5,latlab=limc(3)+5,londel=10.,latdel=10.,title=noms


nbcol=!D.table_size-1
pas =0.16
if noms eq 'daealt.dat' then pas =0.22
if noms eq 'auralt.dat' then pas =0.2
if noms eq 'goralt.dat' then pas =0.17
nbf=0
maxi=max(tab(6,*))
mini=min(tab(6,where(tab(5,*) ge 0)))
ran=maxi-mini

for j=0,numspec-1 do begin
 if tab(5,j) gt 0 then begin
  nbf=nbf+1
  lon(0)=tab(3,j)+pas
  lon(1)=tab(3,j)-pas
  lon(2)=tab(3,j)-pas
  lon(3)=tab(3,j)+pas
  lat(0)=tab(2,j)+pas
  lat(1)=tab(2,j)+pas
  lat(2)=tab(2,j)-pas
  lat(3)=tab(2,j)-pas
  col=tab(6,j)
  col=(col-mini)*nbcol/ran + 1
  if col ge nbcol+1. then col=col-1
  polyfill,lon,lat,color=col
 endif
endfor
print, 'Nb of resampled pixels:', nbf
print, 'Max alti:', maxi, '  Min alti:', mini

;close, /device
;set_plot, 'x'

end
