FUNCTION ISM_sess, fen, error

;+
; NAME
; ISM_sess

;	PURPOSE
; Defines session parameters for ISM sessions (both Mars and Phobos). 
; Does not attempt to read the information from the PDS labels.

; CALLING SEQUENCE:
;	Result = ISM_sess (fen, Error)

; INPUT PARAMETERS:
;	fen: string, nickname of the session

; OUTPUT PARAMETERS:
;	Param: structure containing session parameters:
;	{parameter, lim: IntArr(4)           ; Limits in latitude and longitude
;           , nLines: 0, nSamples: 0   ; Number of lines and samples
;           , nElt: 0                  ; Number of calibrated spectra
;           , DataFile: ''             ; Name of the data file (even channels)
;           , CoorFile: ''             ; Name of the coordinate file
;           , ID: ''                   ; Session acronym  
;           , Region: ''               ; Explicit region name  
;           , date: ''                 ; Date of observation
;	          , IntegTime: 0.            ; Integration time in seconds
;           , Temp: 0.                 ; Average Focal plane temperature (Celsius)
;           , Gain:0}                  ; Analogic gain
;
;	Error: Error flag
;
; MODIFICATION HISTORY:
;	Stephane Erard, july 1998
;-
;******************************************************************************



param = {parameters, Lim: IntArr(4), nLines: 0, nSamples: 0, nElt: 0, $ 
DataFile: '', CoorFile: '', ID: '', Region: '', Date: '', IntegTime: 0.,$
Temp: 0., Gain: 0}

si=size(fen)
if si(si(0)+1) ne 7 then BEGIN 
Message, /CONTINUE, 'Uncorrect session ID: ' + fen
error = 1
RETURN, 0
ENDIF

fen=STRUPCASE(fen)
Param.ID=fen
Param.DataFile=fen+'EVEN.CAL'     ; Default is even channels, calibrated data
Param.CoorFile=fen+'_COOR.DAT'
ISMdef                            ; Defines ISM root directory

CASE fen of 
'PAV': BEGIN
 Param.Region='Pavonis Mons'
 Param.lim=[-5,-125,5,-95]
 Param.nLines=296
 Param.nSamples=8
 Param.nElt=2365
 Param.date='08/02/1989'
 Param.IntegTime=0.125
 Param.Temp=-71.2
 Param.Gain=1
END
'BIB': BEGIN
 Param.Region='Biblis Patera'
 Param.lim=[-5,-145,5,-115]
 Param.nLines=274
 Param.nSamples=8
 Param.nElt=2189
 Param.date='11/02/1989'
 Param.IntegTime=0.125
 Param.Temp=-70.4
 Param.Gain=2
END
'ARA': BEGIN
 Param.Region='Arabia'
 Param.lim=[-10,-40,20,50]
 Param.nLines=98
 Param.nSamples=25
 Param.nElt=2432
 Param.date='21/02/1989'
 Param.IntegTime=0.5
 Param.Temp=-71.2
 Param.Gain=2
END
'DAE': BEGIN
 Param.Region='Daedalia'
 Param.lim=[-30,-180,0,-90]
 Param.nLines=121
 Param.nSamples=25
 Param.nElt=3013
 Param.date='27/02/1989'
 Param.IntegTime=0.5
 Param.Temp=-75.8
 Param.Gain=2
END
'SYR': BEGIN
 Param.Region='Syrtis-Isidis'
 Param.lim=[-10,40,20,130]
 Param.nLines=121
 Param.nSamples=25
 Param.nElt=3005
 Param.date='01/03/1989'
 Param.IntegTime=0.5
 Param.Temp=-76.5
 Param.Gain=2
END
'VMC': BEGIN
 Param.Region='Central Valles Marineris'
 Param.lim=[-20,-140,10,-50]
 Param.nLines=121
 Param.nSamples=25
 Param.nElt=3008
 Param.date='07/03/1989'
 Param.IntegTime=0.5
 Param.Temp=-77.0
 Param.Gain=1
END
'AUR': BEGIN
 Param.Region='Aurorae'
 Param.lim=[-20,-100,10,-10]
 Param.nLines=121
 Param.nSamples=25
 Param.nElt=3007
 Param.date='12/03/1989'
 Param.IntegTime=0.5
 Param.Temp=-76.2
 Param.Gain=2
END
'OLY': BEGIN
 Param.Region='Olympus Mons'
 Param.lim=[0,-180,30,-90]
 Param.nLines=91
 Param.nSamples=25
 Param.nElt=2275
 Param.date='13/03/1989'
 Param.IntegTime=0.5
 Param.Temp=-77.2
 Param.Gain=2
END
'GOR': BEGIN
 Param.Region='Gordii'
 Param.lim=[-10,-180,20,-90]
 Param.nLines=120
 Param.nSamples=25
 Param.nElt=3000
 Param.date='14/03/1989'
 Param.IntegTime=0.5
 Param.Temp=-77.4
 Param.Gain=2
END
'ASC': BEGIN
 Param.Region='Ascraeus Mons'
 Param.lim=[-10,-140,20,-50]
 Param.nLines=120
 Param.nSamples=25
 Param.nElt=2999
 Param.date='21/03/1989'
 Param.IntegTime=0.5
 Param.Temp=-77.0
 Param.Gain=3
END
'HEB': BEGIN
 Param.Region='Hebes'
 Param.lim=[-10,-140,20,-50]
 Param.nLines=120
 Param.nSamples=25
 Param.nElt=2983
 Param.date='26/03/1989'
 Param.IntegTime=0.5
 Param.Temp=-77.2
 Param.Gain=2
END
'PHO': BEGIN
 Param.Region='Phobos session'
 Param.lim=[-10,60,70,180]
 Param.nLines=24
 Param.nSamples=25
 Param.nElt=600
 Param.date='25/03/1989'
 Param.IntegTime=0.5
 Param.Temp=-76.7
 Param.Gain=3
END
ELSE: BEGIN
Message, /CONTINUE, 'Unknown session ID: ' + fen
error = 1
RETURN, 0
ENDELSE
ENDCASE

error = 0
RETURN, param
END
