PRO ISM_raw, Fen, canal0, ODD=odd, PS=ps

;+
; NAME
; ISM_raw

; PURPOSE
; Reads ISM edited (raw) data files and plots spectra in sequence. Also 
; plots an image of the session using grid coordinates (not 
; latitudes/longitudes).

; CALLING SEQUENCE:
;	ISM_raw, fen, canal0

; INPUT PARAMETERS:
;	Fen: string, nickname of the session.
; 	canal0: channel to map (from 1 to 64, number in the even or odd array).

; OUTPUT PARAMETERS:
;	none

; KEYWORD PARAMETERS:
;	ODD: displays odd channels (default is even channels)
;	PS: plots the spectra in a PostScript file

; SUBROUTINES:
; Make calls to ISM_sess, ISM_lam, and ISM_data

; MODIFICATION HISTORY:
;	Stephane Erard, july 1998
;-
;******************************************************************************


si=size(fen)
if si(si(0)+1) ne 7 then begin 
	fen='   '
	print, 'Please give the acronym of the session (without quotes)'
	read, format='(A3)',fen
	print, '  ...and the number of the channel to map (1-64)'
	read, canal0
endif
fen=STRUPCASE(fen)
canal= canal0-1 ; IDL indices

th=2.
smu='!4l!X'
if keyword_set(ps) then begin
 devori=!D.name
 set_plot,'ps'
 device,filename='ISM_spectra.ps',/landscape,/helvetica
 oldfont=!P.font
 !P.font=0
 th=4.
 smu='!Mm!X'
endif

param=ISM_sess(fen, error)     ; read session parameters
IF (error NE 0) THEN RETURN

IF Keyword_Set (odd) THEN BEGIN
 Param.DataFile= Param.ID+'ODD.EDT'
 lam=ISM_lam(error,/ODD)
ENDIF ELSE BEGIN
 Param.DataFile= Param.ID+'EVEN.EDT'
 lam=ISM_lam(error)             ; read wavelength file
ENDELSE
IF (error NE 0) THEN RETURN


tab=ISM_data(param, error)     ; read edited data
IF (error NE 0) THEN RETURN

ShortLam=lam(0:31)          ; split wavelengths to plot separately
LongLam=lam(32:63)

yr=[0.,4000]
IF (fen eq 'PHO') THEN yr=[0.,500]
plot, [0.7,3.3], yr,/nodata,Title=Param.region, xtitle=$
 'Channel number', ytitle='Instrument output (DN)'

 ; Plots a rough map of intensity in channel canal
IF not(keyword_set(ps)) then BEGIN
 ima=intarr(Param.nLines,Param.nSamples,/nozero)
 x=tab.Line-1
 y=Param.nSamples+1-tab.Sample
 ima(x,y)=tab.Spec(canal)
 tvscl, ima > min(ima(where(ima ne 0)))
 print, format='("Image of channel ",I0," (Wavelength = ",F8.4,")")'$
 ,canal+1,lam(canal)
ENDIF

print, format='("The image is ",I3," x ",I2)',Param.nLines, Param.nSamples
print, format='("lines range from  ",I3," to ",I3)',1,Param.nLines
print, format='("samples range from  ",I3," to ",I3)',2,Param.nSamples+1

print, 'Hit line=0 to exit.'
print, 'line/sample ?'
read, l, s
if l gt Param.nLines or s gt Param.nSamples+1 or l lt 1 or s lt 2 then l=0
ref= (l-1)*Param.nSamples+(s-2)   ; accounts for shifts in the array
                                  ; and reverse vertical order (north on top)

; Plots spectra in sequence (reflectance vs wavelength)

WHILE l ne 0 do begin   
 spec2=Tab(ref).spec(0:31)    
 spec1=Tab(ref).spec(32:63)
 oplot, ShortLam, spec2
 oplot, LongLam, spec1
 print, 'line #:',fix(l)
 print, 'sample #:',fix(s)
 print, 'Time:',Tab(ref).hour,':',Tab(ref).minute,':',Tab(ref).second,$
   ',',Tab(ref).tic*125
 print, 'Temperature:',Tab(ref).Temp/100.
 print, 'Mirror angle:',Tab(ref).Mirror
 print, ' '
 print, 'line/sample ?'
 read, l, s
 if l gt Param.nLines or s gt Param.nSamples+1 or l lt 1 or s lt 2 then l=0
 ref= (l-1)*Param.nSamples+(s-2)
ENDWHILE

if keyword_set(ps) then begin
 device, /close
 set_plot,devori
 !P.font=oldfont
endif


END
