PRO ISM_map, Fen, canal0, param, tab, coord, lam, ODD=ODD, PS=ps, SILENT=silent, $
  ALBEDO= albedo, ANGLE=angle, ATM=atm

;+
; NAME
; ISM_map

; PURPOSE
; Reads ISM calibrated data files and maps a complete session in geographic
; coordinates (Mercator projection). 
; Radiance factors (measurements) are normally plotted. Can optionally 
; estimate geometric albedo from reflectance by applying a Minnaert correction
; with exponent k = 0.7 (Mars) or k = 0.55 (Phobos) at all wavelengths, 
; phases and reflectance values.

; CALLING SEQUENCE:
;	ISM_map, Fen, canal0

; INPUT PARAMETERS:
; Fen: string, nickname of the session.
; canal0: channel to map (from 1 to 64, number in the even or odd array).

; OUTPUT PARAMETERS:
; Param: parameters structure for the session
; Tab: data structure for the session
; Coord: coordinate structure for the session
; Lam: wavelength table
;  Output is meant only for ISMcom (widget interface).

; KEYWORD PARAMETERS:
; ODD: displays odd channels (default is even channels)
; ATM: displays atmosphere-corrected files
; ALBEDO: Performs a Minnaert correction to computes and displays albedo.
; ANGLE: permits to plot phase, emergence or incidence
; PS: writes a PostScript file with the map
; SILENT: inhibits messages (default)

; SUBROUTINES:
; Make calls to ISM_sess, ISM_lam, ISM_data, and ISM_coor

; LIMITATIONS:
; The map is ploted using the terrestrial convention for longitudes,
; due to limitations in MAP_SET. Longitude should range from 0 to 360 degrees 
; westward.

; MODIFICATION HISTORY:
;	Stephane Erard, july 1998

;-
;******************************************************************************


;t0= systime(1)
si=size(fen)
if si(si(0)+1) ne 7 then begin 
 fen='   '
 print, 'Please give the acronym of the session (without quotes)'
 read, format='(A3)',fen
 print, '  ...and the number of the channel to map (1-64)'
 read, canal0
endif
fen=STRUPCASE(fen)
canal = canal0 - 1	; IDL indices

if n_elements(silent) eq 0 then silent=1
param=ISM_sess(fen, error)     ; read session parameters
IF (error NE 0) THEN RETURN

if keyword_set(ps) then begin
 devori=!D.name
 set_plot,'ps'
 device,filename='ISM_map.ps',/landscape,/helvetica
 oldfont=!P.font
 !P.font=0
endif


IF Keyword_Set (odd) THEN BEGIN
 Param.DataFile= Param.ID+'ODD.CAL'
 lam=ISM_lam(error,/ODD)
ENDIF ELSE lam=ISM_lam(error)             ; read wavelength file
IF (error NE 0) THEN RETURN
IF Keyword_Set(atm) THEN Param.DataFile= strmid(Param.DataFile,0,8)+'ATM'


tab=ISM_data(param, error)     ; read calibrated data
IF (error NE 0) THEN RETURN
coord=ISM_coor(param,error)    ; read coordinates file
IF (error NE 0) THEN RETURN


if (!D.window eq -1) and ((!D.FLAGS and 256) ne 0) then window
posmap=[.01,.4,0.91,.4+.9/3*!D.x_size/!D.y_size]
latll=param.lim(3)+5
k = 0.7     ; Minnaert's exponent
if (fen eq 'PHO') then begin
 latll=55.
 posmap=[.09,.1,.99,.1+.9/120.*80*!D.x_size/!D.y_size]
 k = 0.55     ; Minnaert's exponent for Phobos
endif
tit1='Reflectance'
if keyword_set(albedo) then tit1= 'Albedo'
titre=Param.region+'!C'+tit1+' map '+string(lam(canal), format='(F8.3)')+' micron'  


ind = where(coord.mirror ne 0)     ; pixels with data
if N_elements(angle) eq 0 then angle=" "
angle = strlowcase(angle)
Case ANGLE of 
'emergence': begin
  maxi=max(Coord(ind).emergence, min=mini)
  titre=Param.region+'!C'+'Map of emergence angles'
  if silent eq 1 then $
    print, 'Min/max emergence angles (degrees):', [mini, maxi]/100.
  ima=bytscl(Coord.emergence, min=mini, max=maxi, top = (!D.N_Colors - 2))
  end
'incidence': begin
  maxi=max(Coord(ind).incidence, min=mini)
  titre=Param.region+'!C'+'Map of incidence angles'
  if silent eq 1 then $
    print, 'Min/max incidence angles (degrees):', [mini, maxi]/100.
  ima=bytscl(Coord.incidence, min=mini, max=maxi, top = (!D.N_Colors - 2))
  end
'phase': begin
  maxi=max(Coord(ind).phase, min=mini)
  titre=Param.region+'!C'+'Map of phase angles'
  if silent eq 1 then $
    print, 'Min/max phase angles (degrees):',[mini, maxi]/100.
  ima=bytscl(Coord.phase, min=mini, max=maxi, top = (!D.N_Colors - 2))
  end 
else: begin
  par = 'even'
  if keyword_set(odd) then par ='odd'
  print, format='("Map of '+par+' channel ",I0," (Wavelength = ",F8.4," micron)")'$
    ,canal+1,lam(canal)
  if keyword_set(albedo) then begin
   print, format=$
   '("Map of geometric albedo tentatively computed with Minnaert''s model (k = ",F4.2,")")',k
   Ima = float(Tab.spec(canal))
   Coord.incidence=coord.incidence < 8500	; for Phobos
   ima = ima*cos(Coord.emergence*(!dtor/100.))^(1.-k)/cos(Coord.incidence*(!dtor/100.))^k
   maxi = max(ima(ind), min=mini)
   ima = bytscl(ima, min=mini, max=maxi, top = (!D.N_Colors - 2))
  endif else begin
   maxi=max(Tab(ind).spec(canal), min=mini)
   ima=bytscl(Tab.spec(canal), min=mini, max=maxi, top = (!D.N_Colors - 2))
  endelse
  if silent eq 1 then $
    print, 'Min/max reflectance:', [mini, maxi] /32768.*.5
  end
endcase

map_set,limit=Param.lim,position=posmap,/mercator,/grid,/label,$
 lonlab=Param.lim(0)-5,latlab=latll,londel=10.,latdel=10., $
 title=titre

if silent eq 1 then print, 'Missing pixels:'
lon=(lat=fltarr(4))
nbf=0

for j=0,Param.nLines*Param.nSamples-1 do begin
if (coord(j).mirror gt 0) then begin
 nbf=nbf+1
 for i=0,3 do begin
  lon(i)=coord(j).coord(2*i)/100.
  if (fen eq 'PHO' and lon(i) lt 0.) then lon(i)=180.
  lat(i)=coord(j).coord(2*i+1)/100.
 endfor 
 col = ima(j)
 if (min(lon) ne 0.) then polyfill,lon,lat,color=col  ; security only for Phobos
endif else if silent eq 1 then print, j
endfor
if silent eq 1 then print, 'nb of spectra:', nbf


if keyword_set(ps) then begin
 device, /close
 set_plot,devori
 !P.font=oldfont
endif
;print, systime(1)-t0

END
