;+
; NAME
; ISMcom

;	PURPOSE
; Widget interface to the ISM data base. 
; Plot reflectance or albedo maps, and plot spectra for every main session 
;  including Phobos.

; CALLING SEQUENCE:
;	ISMcom

; INPUT PARAMETERS:
;	none

; OUTPUT PARAMETERS:
; widgets

; SUBROUTINES:
; Make calls to ISM_map, which calls ISM_sess, ISM_lam, ISM_data, and ISM_coor

; COMMON BLOCKS:
; ISM_com, ISM_com3: internal variables.
; ISM_com2: sessions parameters, data, and coordinates + wavelengths.

; LIMITATIONS:
; The map is ploted using the terrestrial convention for longitudes,
; due to limitations in MAP_SET. Longitude should range from 0 to 360 degrees 
; westward.

; MODIFICATION HISTORY:
;	July 1998: written, Stephane Erard (erard@ias.fr)
; November 98: added spectra extraction in a separated window.
;
;-
;******************************************************************************



PRO Plotspec, l, s, albedo=albedo

common ISM_com2, param, tab, coord, lam

ShortLam=lam(0:31)          ; split wavelengths to plot separately both orders
LongLam=lam(32:63)

if l gt Param.nLines or s gt Param.nSamples+1 or l lt 1 or s lt 2 then l=0
ref= (l-1)*Param.nSamples+(s-2)   ; accounts for shifts in the array
                                  ; and reverse vertical order (north on top)
spec2=Tab(ref).spec(0:31)/32768.*.5     ; convert to reflectance    
spec1=Tab(ref).spec(32:63)/32768.*.5
print, 'line #:',fix(tab(ref).line)
print, 'sample #:',fix(tab(ref).sample)
IF Tab(ref).Mirror gt 0 then begin
 if keyword_set(albedo) then begin 
  k = 0.7     ; Minnaert's exponent
  if param.ID eq 'PHO' then k=0.55
  Coord(ref).incidence=coord(ref).incidence < 8500
  coef=cos(Coord(ref).emergence*(!dtor/100.))^(1.-k)/cos(Coord(ref).incidence*(!dtor/100.))^k
; print, coef
  spec2=spec2*coef
  spec1=spec1*coef
 endif
 oplot, ShortLam, spec2
 oplot, LongLam, spec1
 print, format="('  Time:   ',I2,':',I2 ,':',I2,',',I3)",Tab(ref).hour,$
   Tab(ref).minute,Tab(ref).second,Tab(ref).tic*125 
 print, '  Temperature:',Tab(ref).Temp/100.
 print, '  Mirror angle:',Tab(ref).Mirror
 angles=[Coord(ref).incidence,Coord(ref).emergence,Coord(ref).phase]/100.
 print, format="('  Angles (degrees): i=', F5.2,'  e=', F5.2,'  phi=',F5.2)", angles
 print, $
 format="('  Coordinates (for even channels, terrestrial. conv.): Long= ',F7.2,' Lat= ',F7.2)", $
 total(Coord(ref).coord([0,2,4,6])/100.)/4., total(Coord(ref).coord([1,3,5,7])/100.)/4.
ENDIF ELSE BEGIN
 print, '  No measurement'
ENDELSE

end


PRO EtHop2_Event, Event

; Processing of events: only Plot button results in action

;common ISM_com, Sess, chan, par, cal, alb, go, bex, bspec, S0, b0, b2, F1
common ISM_com3, alb, nline, nsample, go , bex, bmap, S2, S3

WIDGET_CONTROL,Event.Id, get_value=val

case event.id of
 go: begin
;   print, nline, nsample, alb
   plotspec, nline, nsample, albedo=alb
   end
 S2 : nline = val
 S3 : nsample = val
 bmap : begin
     WIDGET_CONTROL,event.top,/DESTROY
     ismcom
     end
 bex: WIDGET_CONTROL, event.top, /DESTROY
endcase
;print,event.id

END



PRO ISMcom1, par, albe, GROUP=Group

; This widget selects the spectra to be plotted.


common ISM_com2, param, tab, coord, lam
common ISM_com3, alb, nline, nsample, go, bex, bmap, S2, S3


  IF N_ELEMENTS(Group) EQ 0 THEN GROUP=0
;  junk   = { CW_PDMENU_S, flags:0, name:'' }     ; structure definition


; Intial values for display:
nsample = 2      ; Sample 2
nline = 2        ; Line 2
if n_elements(par) eq 0 then par=0     ; (assume even channels by default)
if n_elements(albe) eq 0 then albe=0     ; (assume reflectance by default)
alb=albe

MAIN12 = WIDGET_BASE(GROUP_LEADER=Group, $     ; palette
      ROW=1, $
      MAP=1, $
      TITLE='ISM spectra', $
      UVALUE='MAIN12')

  window, /free
  smu='!4l!X'
  yr=[0.,0.5]
  IF (param.ID eq 'PHO') THEN yr=[0.,0.1]
  yt='Radiance factors'
  if alb eq 1 then  yt='Geometric albedo'
  yt1=", even channels"
  IF par eq 1 then yt1=", odd channels"
  plot, [0.7,3.3], yr,/nodata,Title=Param.region+yt1, xtitle=$
    'Wavelength ('+smu+'m)', ytitle=yt

  reste=WIDGET_BASE(MAIN12,/column)
  bmap=widget_button(reste, Value='Other session', yO=-30, XS=50, YS= 50)
  go=widget_button(reste, Value='Plot', Uvalue='Plot', yO=-30, XS=50, YS= 50)

  s2=widget_slider(reste,minimum=1, maximum=param.Nlines, value=nline, $
     uvalue=nline, title="Line number")
  s3=widget_slider(reste,minimum=2, maximum=param.NSamples+1, value=nsample, $
     uvalue=nsample, title="Sample number")

  bex=widget_button(reste, Value='Done', XS=20, YS= 20)

  WIDGET_CONTROL, MAIN12, /REALIZE

  XMANAGER, 'EtHop2', MAIN12


END



PRO EtHop_Event, Event

; Processing of session events: only Plot button results in action

common ISM_com, Sess, chan, par, cal, alb, go, bex, bspec, S0, b0, b2, F1
common ISM_com2, param, tab, coord, lam
Fen=['PAV','BIB','ARA','DAE','SYR','VMC','AUR','OLY','GOR','ASC','HEB','PHO']

WIDGET_CONTROL,Event.Id, get_value=val

case event.id of
 go: begin
;   print,  Sess, chan, par, cal, alb
 ISM_map, Sess, chan,  param, tab, coord, lam, odd=par, albedo=alb
   end
 F1: Sess = Fen(val)
 S0 : chan = val
 B0 : par = val
 B2 : alb = val
 bspec : begin
     sz=size(param)
     if sz(0) eq 0 then begin
      Print,'  Select a session first'
     endif else begin
      WIDGET_CONTROL,event.top,/DESTROY
      ISMcom1, par, alb
     endelse
     end
 bex: WIDGET_CONTROL, event.top, /DESTROY
endcase
;print,event.id

END



PRO ISMcom, GROUP=Group

; Main widget: select session and plot session map.

common ISM_com, Sess, chan, par, cal, alb, go, bex, bspec, S0, b0, b2, F1


  IF N_ELEMENTS(Group) EQ 0 THEN GROUP=0
;  junk   = { CW_PDMENU_S, flags:0, name:'' }     ; structure definition


; Intial values for display:
chan = 12      ; Channel 12 even (or odd), ie 24 or 23
par = 0        ; even channels
cal = 0        ; calibrated data
alb = 0        ; in reflectance
Sess ='SYR'    ; Syrtis session
nSes = 4


MAIN13 = WIDGET_BASE(GROUP_LEADER=Group, $     ; palette
      ROW=1, $
      MAP=1, $
      TITLE='ISM observation sessions', $
      UVALUE='MAIN13')

Session=['Pavonis','Biblis','Arabia','Daedalia','Syrtis-Isidis','V. Marineris','Aurorae', $
  'Olympus','Gordii','Ascraeus','Hebes','Phobos']
Fen=['PAV','BIB','ARA','DAE','SYR','VMC','AUR','OLY','GOR','ASC','HEB','PHO']
  F1=cw_bgroup(MAIN13, /Column, /EXCLUSIVE, /NO_REL, $
	     Session, UVALUE=Sess, SET_VALUE=Nses)


  window, /free
  reste=WIDGET_BASE(MAIN13,/column)
  bspec=widget_button(reste, Value='Extract Spectra', yO=-30, XS=50, YS= 50)
  go=widget_button(reste, Value='Plot', Uvalue='Plot', yO=-30, XS=50, YS= 50)
 
  reste2=WIDGET_BASE(reste,/row)
  b0=cw_bgroup(reste2, /Column, /EXCLUSIVE, /NO_REL, $
	     ['Even', 'Odd'], UVALUE=par, SET_VALUE=par)
  b2=cw_bgroup(reste2, /Column, /EXCLUSIVE, /NO_REL, $
	     ['Reflectance', 'Albedo'], UVALUE=alb, SET_VALUE=alb)

  s0=widget_slider(reste,minimum=1, maximum=64, value=chan, uvalue=chan)

;  b1=cw_bgroup(reste, /Column, /EXCLUSIVE, /NO_REL, $
;	     ['Calibrated', 'Raw'], UVALUE=cal)

  bex=widget_button(reste, Value='Done',  XS=20, YS= 50)

  WIDGET_CONTROL, MAIN13, /REALIZE

  XMANAGER, 'EtHop', MAIN13


END
