MODULE MZDIST
  
  USE CONSTANTS
  IMPLICIT NONE

  PRIVATE
  PUBLIC :: ZDIST

CONTAINS

!----------------------------------------------------------------------------

  SUBROUTINE ZDIST (nt, ns, a, zav, sig_z, jpel, hpel, hspel, jgg, cgg)
! 
! computes the charge distribution of dust grains
! Kimura+2016 or Weingartner & Draine 2001 (WD) formalism
! van Hoof et al 04 revision for Emin (also Weingartner et al 06, W06)
!

  ! modules with global variables
    USE CONSTANTS
    USE UTILITY

    IMPLICIT NONE
 
    ! arguments
    INTEGER,        INTENT (IN)    :: nt, ns                       ! index of grain type and size
    REAL (KIND=dp), INTENT (IN)    :: a                            ! grain radius (cm)
    REAL (KIND=dp), INTENT (OUT)   :: zav, sig_z                   ! mean charge and standard deviation
    REAL (KIND=dp), INTENT(OUT)    :: jpel, hpel, hspel, jgg, cgg  ! PE rate and heating, gas-grain rates and cooling

    ! local variables
    INTEGER                     :: i, k, imin, nzbi, nzbl
    REAL (KIND=dp)              :: uait, tmp                       ! autoionization threshold
    REAL (KIND=dp)              :: aa, zqa, zqb, zq, zc, jj, hh, hs, xqe2a, le
    REAL (KIND=dp)              :: z1(1), je(1), tt(1), z2(2)
    REAL (KIND=dp), ALLOCATABLE :: zbl(:), fzl(:), jp(:), jm(:), cp(:), cm(:)


    aa = a/1.0e-8_dp   ! cm to Angstroems

    ! min and max charges
    uait = -(p_uait(nt,1) + p_uait(nt,2)*aa + p_uait(nt,3)/aa)
    zmin = DBLE( FLOOR(uait/14.4_dp*aa) + 1 )
    zmax = DBLE( FLOOR( ((hnumax/everg-wf(nt))/14.4_dp*aa + 0.5_dp - 0.3_dp/aa) / (1.0_dp + 0.3_dp/aa) ) )

    ! first estimate Zeq solution of Jpe+Jp=Jm
    zqa = zmin
    zqb = zmax
    CALL GET_ZEQ (nt, ns, zqa, zqb, aa)
    zeq = (zqa+zqb)/2.0_dp 
    zq = DBLE( NINT(zeq) )

    ! get nr of Z bins and Z grid
    nzbi = nz_bg + NINT( DBLE(nz_sg-nz_bg)/(ABS(zq)/ztrans + 1.0_dp) )
    nzbl = MIN( nzbi, NINT(ABS(zmax-zmin))+1)
    IF (nzbl /= nzbi) THEN  ! get the charge midpoint
       ! use the full Z range (small grains)
       zc = DBLE(NINT((zmin+zmax)/2.0_dp))
       DO i=1,nzbl 
          tmp = zmin + DBLE(i-1)
          IF (tmp <= zmax) k = i
       ENDDO
       nzbl = k
       ALLOCATE (zbl(nzbl))
       DO i=1,nzbl
          zbl(i) = zmin + DBLE(i-1)
       ENDDO
    ELSE
       ! use a range around Zeq (big grains)
       zc = zq
       ALLOCATE (zbl(nzbl))
       DO i=1,nzbl
          zbl(i) = zc + DBLE(i-nzbl/2-1)
       ENDDO
    ENDIF
    ALLOCATE(fzl(nzbl),jp(nzbl),jm(nzbl),cp(nzbl),cm(nzbl))

    ! get full charge distribution
    fzl = 0.0_dp
    fzl(nzbl/2+1) = 1.0_dp ! start value @ z = zc
    imin = 0
    DO i=nzbl/2+1, 2, -1
       z2 = zbl(i-1:i)
       CALL PE_CURRENT( nt, ns, z2(1), aa, jpel, hpel, hspel, le )
       CALL GAS_CURRENTS (nt, 2, 1, nion, z2, aa, t_gas, le, jp, jm, cp, cm)
       tmp = jm(2) / (jpel + jp(1))
       IF ( ((tmp>istiny) .AND. (tmp<1.0_dp/istiny)).AND. ((fzl(i)>tiniest/istiny) .AND. (fzl(i)<istiny/tiniest)) )THEN 
          fzl(i-1) = fzl(i) * tmp
       ELSE 
          fzl(i-1) = 0.0_dp
       ENDIF
       IF (zbl(i-1)==zmin) imin = i-1
    ENDDO
    DO i=nzbl/2+1,nzbl-1
       z2 = zbl(i:i+1)
       CALL PE_CURRENT( nt, ns, z2(1), aa, jpel, hpel, hspel, le )
       CALL GAS_CURRENTS (nt, 2, 1, nion, z2, aa, t_gas, le, jp, jm, cp, cm)
       tmp = (jpel + jp(1)) / jm(2)
       IF ( ((tmp>istiny) .AND. (tmp<1.0_dp/istiny)).AND. ((fzl(i)>tiniest/istiny) .AND. (fzl(i)<istiny/tiniest)) )THEN 
          fzl(i+1) = fzl(i) * tmp
       ELSE 
          fzl(i+1) = 0.0_dp
       ENDIF
    ENDDO

    ! build the final charge distribution
    fzl = fzl / SUM(fzl)
    nzb = 0
    DO i=1, nzbl
       IF ((fzl(i)>fzmin) .AND. (fzl(i)<=1.0_dp)) nzb = nzb + 1
    ENDDO
    DEALLOCATE (jp,jm,cp,cm)
    IF (nzb>0) THEN 
       ALLOCATE (zb(nzb),fz(nzb)) 
       ALLOCATE (jp(nzb),jm(nzb),cp(nzb),cm(nzb))
       k = 0
       DO i=1, nzbl
          IF ((fzl(i)>fzmin) .AND. (fzl(i)<=1.0_dp)) THEN 
             k = k + 1
             zb(k) = zbl(i)
             fz(k) = fzl(i)
          ENDIF
       ENDDO
       fz = fz / SUM(fz)

       ! compute the average charge and final rates
       zav = SUM(zb*fz)                       ! mean charge 
       sig_z = SQRT(SUM(zb**2*fz)-zav**2)     ! variance of fz 
       jpel = 0.0_dp
       hpel = 0.0_dp
       hspel = 0.0_dp
       jgg = 0.0_dp
       cgg = 0.0_dp
       DO i = 1, nzb
          z1(1) = zb(i)
          CALL PE_CURRENT( nt, ns, z1(1), aa, jj, hh, hs, le)
          jpel = jpel + jj*fz(i)
          hpel = hpel + hh*fz(i)
          hspel = hspel + hs*fz(i)
          CALL GAS_CURRENTS (nt, 1, 1, nion, z1, aa, t_gas, le, jp, jm, cp, cm)
          jgg = jgg + (jp(1) + jm(1))*fz(i)
          cgg = cgg + (cp(1) + cm(1))*fz(i)
       ENDDO

       IF (imin > 0) THEN
          z1(1) = zmin
          CALL PE_CURRENT( nt, ns, zmin, aa, jj, hh, hs, le)
          CALL GAS_CURRENTS (nt, 1, 1, nion, z1, aa, t_gas, le, tt, je, tt, tt)
          xqe2a = xqe2/(1.e-8_dp*aa)
          cgg = cgg + fzl(imin)*je(1)*( (wf(nt)-ebg(nt,ns))*everg + (zmin-5e-1_dp)*xqe2a )
          IF (INDEX(t_opt(nt),'WD')>0) cgg = cgg - fzl(imin)*je(1)*((p_ea(nt,1)/(aa+p_ea(nt,2)) )*xqe2a)
       ENDIF

    ELSE 
       WRITE (*,*) ''
       WRITE (*,*) '(F) DM_zdist/ZDIST: f(Z) has no values above ',fzmin
       WRITE (*,*) ' grain type= ',gtype(nt),' a (nm)= ',aa
       WRITE (*,*) ''
       STOP
    ENDIF

  END SUBROUTINE ZDIST

!----------------------------------------------------------------------------
  SUBROUTINE GET_ZEQ (nt, ns, z_a, z_b, a)
  ! finds equilibirum charge from Jpe + Jp = Jm

    USE CONSTANTS
    USE UTILITY

    IMPLICIT NONE

    INTEGER, INTENT (IN)           :: nt               ! index of grain type
    INTEGER, INTENT (IN)           :: ns               ! index of grain size
    REAL (KIND=dp), INTENT (INOUT) :: z_a, z_b
    REAL (KIND=dp), INTENT (IN)    :: a                ! grain size

    INTEGER                        :: i
    REAL (KIND=dp)                 :: z, fa, fb, f, je, hh, hs, le
    REAL (KIND=dp)                 :: z1(1), jp(1), jm(1), cp(1), cm(1)

    z1(1) = z_a
    CALL PE_CURRENT ( nt, ns, z_a, a, je, hh, hs, le )
    CALL GAS_CURRENTS (nt, 1, 1, nion, z1, a, t_gas, le, jp, jm, cp, cm)
    fa = je + jp(1) - jm(1)

    z1(1) = z_b
    CALL PE_CURRENT ( nt, ns, z_b, a, je, hh, hs, le )
    CALL GAS_CURRENTS (nt, 1, 1, nion, z1, a, t_gas, le, jp, jm, cp, cm)
    fb = je + jp(1) - jm(1)
    
    i = 0
    IF (fa*fb > 0.0_dp) THEN
       PRINT *, "  (W) GET_ZEQ:  Wrong initial guess"
       PRINT *, "      Za = ", z_a, "  fa = ", fa
       PRINT *, "      Zb = ", z_b, "  fb = ", fb
    ENDIF
    DO
       z = 0.5_dp * (z_a + z_b)
       z1(1) = z
       CALL PE_CURRENT ( nt, ns, z, a, je, hh, hs, le )
       CALL GAS_CURRENTS (nt, 1, 1, nion, z1, a, t_gas, le, jp, jm, cp, cm)
       f = je + jp(1) - jm(1)
       IF (f*fa > 0.0_dp) THEN
          z_a = z
          fa = f
       ELSE
          z_b = z
          fb = f
       ENDIF
       i = i + 1
       IF ((z_b-z_a) < 5.0e-1_dp) EXIT
    ENDDO
    
END SUBROUTINE GET_ZEQ

!----------------------------------------------------------------------------

SUBROUTINE GAS_CURRENTS (nt, nz, i1, i2, zg, ag, tgas, le, jpi, jmi, cpi, cmi)
! computes the gas current (s-1) from ions or electrons for the charge distribution

    USE CONSTANTS
    USE UTILITY

    IMPLICIT NONE

    INTEGER, INTENT (IN)         :: nt, nz           ! index for grain type, nr of charge bins
    INTEGER, INTENT (IN)         :: i1, i2           ! min and max indices to select ions
    REAL (KIND=dp), INTENT (IN)  :: ag, zg(nz)       ! size (Angstroems) and charge grid of grain (unit of e)
    REAL (KIND=dp), INTENT (IN)  :: tgas, le         ! gas temperature, electron mean free path
    REAL (KIND=dp), INTENT (OUT) :: jpi(nz), jmi(nz) ! rates for positive and negative currents
    REAL (KIND=dp), INTENT (OUT) :: cpi(nz), cmi(nz) ! cooling rates for positive and negative currents

    INTEGER                      :: k
    REAL (KIND=dp)               :: mii, tau
    REAL (KIND=dp)               :: nu(nz), theta(nz), jt(nz), ct(nz), stick(nz)

    jpi = 0.0_dp
    jmi = 0.0_dp
    cpi = 0.0_dp
    cmi = 0.0_dp

    ! ion list in GAS.DAT
    DO k = i1, i2
       tau = (ag*1e-8_dp) * xkb*tgas / xqe2/zi(k)**2
       nu = zg/zi(k)
       mii = mi(k)*amu 

       ! get Jtilde and lambda_tilde function from Draine & Sutin 87
       WHERE (nu<0.0_dp) 
          jt = (1.0_dp-nu/tau) * ( 1.0_dp + SQRT(2.0_dp/(tau-2.0_dp*nu)) )
          ct = (2.0_dp-nu/tau) * ( 1.0_dp + 1.0_dp/SQRT(tau-nu) ) 
       ENDWHERE
       WHERE (nu==0.0_dp) 
          jt = 1.0_dp + SQRT(xpi/2.0_dp/tau)
          ct = 2.0_dp + 1.5_dp*SQRT(xpi/2.0_dp/tau)
       ENDWHERE
       WHERE (nu>0.0_dp)
          theta = nu / ( 1.0_dp + 1.0_dp/SQRT(nu) )
          jt = ( 1.0_dp + 1.0_dp/SQRT(4.0_dp*tau + 3.0_dp*nu) )**2 * EXP(-theta/tau)
          ct = (2.0_dp+nu/tau)*( 1.0_dp + 1.0_dp/SQRT(3.0_dp/2.0_dp/tau + 3.0_dp*nu) )*EXP(-theta/tau) 
       ENDWHERE

       ! get sticking coefficient
       stick = 1.0_dp
       IF ( (zi(k) == -1.0_dp) .AND. (ABS(mii/xme-1.0_dp)-1.0_dp < 1e-2_dp) ) THEN   ! electrons
          stick = p_se(nt,1) * ( 1.0_dp - EXP(-p_se(nt,2)*ag/le) )
          WHERE( (zg < 0.0_dp) .AND. (zg <= zmin) ) stick = 0.0_dp
          if ( p_se(nt,3) > 0.0_dp ) then 
             WHERE( (zg == 0.0_dp) .OR. ((zg < 0.0_dp) .AND. (zg>zmin))) stick = &
                  & stick * ABS(p_se(nt,3)) / ( 1.0_dp + EXP(2e1_dp-4.68e-1_dp*ag**3) ) ! PAH correction Eq.28 of WD01
          endif
       ENDIF

       IF (zi(k) < 0.0_dp) THEN
          jmi = jmi + xpi*(ag*1.0e-8_dp)**2 * stick * iden(k) * SQRT(8.0_dp*xkb*tgas/xpi/mii) * jt
          cmi = cmi + xpi*(ag*1.0e-8_dp)**2 * stick*iden(k)*SQRT(8.0_dp*xkb*tgas/xpi/mii)*xkb*tgas * ct
       ELSE IF (zi(k) > 0.0_dp) THEN
          jpi = jpi + xpi*(ag*1.0e-8_dp)**2 * stick * iden(k) * SQRT(8.0_dp*xkb*tgas/xpi/mii) * jt
          cpi = cpi + xpi*(ag*1.0e-8_dp)**2 * stick*iden(k)*SQRT(8.0_dp*xkb*tgas/xpi/mii)*xkb*tgas * ct
       ENDIF

    ENDDO

  END SUBROUTINE GAS_CURRENTS
  
!----------------------------------------------------------------------------

  SUBROUTINE PE_CURRENT( nt, ns, zg, ag, jp, hp, hsp, le )
    ! computes the photelectric current (s-1) for the charge distribution

    USE CONSTANTS
    USE UTILITY

    IMPLICIT NONE

    REAL (KIND=dp), INTENT (OUT) :: jp, hp, hsp, le

    INTEGER, INTENT (IN)         :: nt, ns          ! index for grain type and size
    REAL (KIND=dp), INTENT (IN)  :: ag, zg          ! size (Angstroems) and charge grid of grain (unit of e)

    INTEGER                      :: i, k, nfrq, jtresh   
    REAL (KIND=dp)               :: jp1, hp1, hsp1
    REAL (KIND=dp)               :: xqe2a, emin, theta, ipdt, ipet
    REAL (KIND=dp), ALLOCATABLE  :: hnu(:), teeta(:), elow(:), ehigh(:), e2(:), et(:), alpha(:), beta(:), tt(:), l_e(:)
    REAL (KIND=dp), ALLOCATABLE  :: y0(:), y1(:), y2(:), yt(:)
    REAL (KIND=dp)               :: jp2, hp2, hsp2
    REAL (KIND=dp), ALLOCATABLE  :: sig_pd(:)

    xqe2a = xqe2/(1.e-8_dp*ag)/everg   ! in eV

    ! get Emin (eV)
    IF (zg < -1.0_dp) THEN 
       theta = ABS(zg+1.0_dp) / ( 1.0_dp + 1.0_dp/SQRT(ABS(zg+1.0_dp)) )
       ! repulsion barrier for tunneling (emin>0) W06 Eq.3 (see also van Hoof et al 04)
       emin = theta*xqe2a * ( 1.0_dp - 3e-1_dp/(ag/10.0_dp)**4.5e-1_dp/(ABS(zg+1.0_dp))**2.6e-1_dp )  
!       emin = ABS(zg+1.0_dp)*xqe2a / (1.0_dp+(27.0_dp/ag)**7.5e-1_dp) ! WD01 eq. 7
    ELSE 
       emin = 0.0_dp
    ENDIF

    ! PE threshhold and frequency grid
    IF (INDEX(t_opt(nt),'WD')>0) THEN
       ! WD01: Eqs. 2 and 6 used for both carbon and silicate grains
       ipet = emin + wf(nt) + (zg+0.5_dp)*xqe2a + (zg+2.0_dp)*(3e-1_dp/ag)*xqe2a
    ELSE
       ! Brus+1988 for dielectrics
       ! NB neglect dielectric correction because epsilon > 10 
       ipet = emin + wf(nt) + (zg+0.5_dp)*xqe2a + p_6(nt,1)*(xpi*p_6(nt,2)/2.0_dp-1.0_dp)*xqe2a/2.0_dp
    ENDIF
    IF ( (1.0_dp-ipet*everg/hnumax) > 1e-4_dp ) THEN ! PE works only if ipet < hnumax
       ALLOCATE (hnu(n_qabs))
       hnu = xhp*freq_qabs/everg
       i = 1
       DO WHILE (hnu(i) <= ipet) 
          i = i+1
       ENDDO
       jtresh = i
       nfrq = jfreqmax-jtresh+1
       DEALLOCATE (hnu)
       ALLOCATE (hnu(nfrq),teeta(nfrq),elow(nfrq),ehigh(nfrq),e2(nfrq),et(nfrq),alpha(nfrq),beta(nfrq),tt(nfrq),l_e(nfrq))
       ALLOCATE(y0(nfrq),y1(nfrq),y2(nfrq),yt(nfrq))
       hnu = xhp*freq_qabs(jtresh:jfreqmax)/everg  ! in eV

       ! photoelectron energies (in eV)
       ehigh = emin + hnu - ipet
       IF (zg >= 0.0_dp) THEN 
          elow = -(zg+1.0_dp)*xqe2a
          teeta = hnu - ipet - elow 
          y2 = ehigh**2 * (ehigh-3.0_dp*elow) / (ehigh-elow)**3 
          e2 = ehigh * (ehigh-2.0_dp*elow) / (ehigh-3.0_dp*elow)/2.0_dp ! mean kinetic energy of electron
       ELSE
          elow = emin
          teeta = hnu - ipet
          y2 = 1.0_dp
          e2 = (ehigh+elow)/2.0_dp
       ENDIF

       IF (p_ke(nt) > 0.0_dp) e2 = p_ke(nt)*teeta ! force mean KE of electron to p_ke fraction of teeta (=hnu-W)
                                                  ! p_ke from >0 to 1: max e2 = teeta or et = hnu (K16 case)
       et = e2 + hnu - teeta ! = e2 + ipet + elow: mean energy of the electron *within* the grain
       
       ! attenuation length
       ! NB qi_abs sorted as freq_qabs (reverse to lamb_qabs) after CALL GET_QEXT
       IF (INDEX(t_opt(nt),'WD')>0) THEN
          beta = 4.0_dp*xpi*(ag*1e-8_dp)*freq_qabs(jtresh:jfreqmax)/clight  ! beta = 2*x (or n2 ~ 1) a proxy to WD01 f(z)
          ! fits for n2 over hnu = 4 - 14 eV
          ! IF (INDEX(gtype(nt),'Sil')>0) beta = ag / 1.61e6/hnu**(-3.67)    
          ! IF (INDEX(gtype(nt),'Gra')>0 .OR. INDEX(gtype(nt),'PAH')>0 .OR. INDEX(gtype(nt),'amCBEx')>0) &
          !    & beta = ag/2.5e3/hnu**(-1.22)
       ELSE
          ! from optical theorem (Krugel, Bohren+Hufman)
          beta = ( qi_abs(nt,ns,jtresh:jfreqmax) + qidiff(nt,ns,jtresh:jfreqmax) ) * 3.0_dp/4.0_dp
       ENDIF
       IF (INDEX(t_opt(nt),'WD')>0) THEN
          l_e = 10.0_dp
       ELSE ! K16
          l_e = 10.0_dp * (p_le(nt,1)*et**p_le(nt,2) + p_le(nt,3)*et**p_le(nt,4)) ! le in Angstroems
       ENDIF
       alpha = beta + ag/l_e

       ! get PE yield
       IF (INDEX(t_opt(nt),'WD')>0) THEN
          ! WD01 yield
          tt = alpha**2 - 2.0_dp*alpha + 2.0_dp*(1.0_dp-EXP(-alpha))
          y1 = (beta/alpha)**2 * tt / ( beta**2 - 2.0_dp*beta + 2.0_dp*(1.0_dp-EXP(-beta)) ) ! Watson enhancement
          y0 = p_y(nt,1)*(teeta/wf(nt))**p_y(nt,3) / (1.0_dp + p_y(nt,2)*(teeta/wf(nt))**p_y(nt,3))
       ELSE
          ! K16 yield (eq A12)
          tt = (1.0_dp + SQRT(1.0_dp-e2/et)) * alpha**2 * (1.0_dp-EXP(-alpha)) &
               & + (alpha**2-2.0_dp*alpha+2.0_dp*(1.0_dp-EXP(-alpha))) * (alpha - beta)       
          y1 = tt * (beta/alpha)**3 / (alpha-beta+1.0_dp) / ( beta**2 - 2.0_dp*beta + 2.0_dp*(1.0_dp-EXP(-beta)) )
          y0 = 0.5_dp * ( 1.0_dp - SQRT(1.0_dp-e2/et) )  ! semi-classical threshold
          ! K16 ref yield (semi-infinite slab)
          !tt = (alpha-beta) / beta
          !y0 = 0.5_dp * ( 1.0_dp - SQRT(1.0_dp-e2/et) + tt*LOG( (SQRT(1.0_dp-e2/et)+tt) / (1.0_dp+tt) ) )
       ENDIF
       DO i = 1,nfrq  ! PE yield must be <= 1
          yt(i) = y2(i) * MIN( y0(i)*y1(i), 1.0_dp )
       ENDDO

       ! get average 1/le
       IF (INDEX(t_opt(nt),'WD')>0) THEN
          le = 10.0_dp
       ELSE
          ! need <1/le> in GAS_CURRENTS for recombination rate
          le = l_e(1)
          IF (NFRQ > 1) THEN 
             tt = yt*qi_abs(nt,ns,jtresh:jfreqmax)*isrfuv(jtresh:jfreqmax)/hnu/l_e
             jp1 = XINTEG2( 1, nfrq, nfrq, freq_qabs(jtresh:jfreqmax), tt )
             tt = yt*qi_abs(nt,ns,jtresh:jfreqmax)*isrfuv(jtresh:jfreqmax)/hnu
             le = jp1 / XINTEG2( 1, nfrq, nfrq, freq_qabs(jtresh:jfreqmax), tt )
             le = 1.0_dp/le
          ENDIF
       ENDIF

       ! rotational excitation (DL98)
       tt = yt*qi_abs(nt,ns,jtresh:jfreqmax)*isrfuv(jtresh:jfreqmax)*(e2+(zg+1.0_dp)*xqe2a)/hnu
       hsp1 = xpi*(ag*1e-8_dp)**2 * XINTEG2( 1, nfrq, nfrq, freq_qabs(jtresh:jfreqmax), tt )
       ! PE heating rate
       tt = yt*qi_abs(nt,ns,jtresh:jfreqmax)*isrfuv(jtresh:jfreqmax)*e2/hnu
       hp1 = xpi*(ag*1e-8_dp)**2 * XINTEG2( 1, nfrq, nfrq, freq_qabs(jtresh:jfreqmax), tt )
       ! PE current
       tt = tt/e2/everg
       jp1 = xpi*(ag*1e-8_dp)**2 * XINTEG2( 1, nfrq, nfrq, freq_qabs(jtresh:jfreqmax), tt )
       
    ELSE   ! no photons of energies above PE threshhold

       jp1 = 0.0_dp
       hp1 = 0.0_dp
       hsp1 = 0.0_dp

    ENDIF

    ! photodetachment part: !!!! hnu_pdt = Emin + EA(Z+1) !!!! 
    IF (INDEX(t_opt(nt),'WD')>0) THEN
       ! Eq. 18 of WD01
       ipdt = emin + wf(nt) - ebg(nt,ns) + (zg+5e-1_dp)*xqe2a - ( p_ea(nt,1)/(ag+p_ea(nt,2)) )*xqe2a
    ELSE
       ! Brus+1988 (neglect dielectric correction)
       ipdt = emin + wf(nt) - ebg(nt,ns) + (zg+0.5_dp)*xqe2a - p_6(nt,1)*(xpi*p_6(nt,2)/2.0_dp-1.0_dp)*xqe2a/2.0_dp
    ENDIF
    IF ((ipdt < hnumax/everg) .AND. (zg < 0.0_dp)) THEN  ! for photons of energies above PD threshhold
       IF (ALLOCATED(hnu)) DEALLOCATE (hnu)
       IF (ALLOCATED(tt)) DEALLOCATE (tt)
       ALLOCATE (hnu(n_qabs))
       hnu = xhp*freq_qabs/everg
       i = 1
       DO WHILE (hnu(i) <= ipdt) 
          i = i+1
       ENDDO
       jtresh = i
       nfrq = jfreqmax-jtresh+1
       DEALLOCATE (hnu)
       ALLOCATE (hnu(nfrq),sig_pd(nfrq),tt(nfrq))
       hnu = xhp*freq_qabs(jtresh:jfreqmax)/everg
       tt = (hnu-ipdt)/s_ea(nt,2)
       sig_pd = ABS(zg)*(2.0_dp*xpi*xqe2*xhp*s_ea(nt,1)/3.0_dp/xme/clight/s_ea(nt,2)/everg) * & 
            & tt/(1.0_dp+tt**2/3.0_dp)**2  ! size independent cross-section (C6F6- value) or (osc. strength) ~ 1/size^2
!       sig_pd = sig_pd * (ag/2.34_dp)**2  ! prop to geom. cross-section, scaled to C6F6- (WD01) constant osc. strength

       ! rotational excitation (DL98)
       ! NB assume max energy for the detached electron
       tt = sig_pd*isrfuv(jtresh:jfreqmax)*(hnu-ipdt+emin+(zg+1.0_dp)*xqe2a)/hnu
       hsp2 = XINTEG2( 1, nfrq, nfrq, freq_qabs(jtresh:jfreqmax), tt )
       ! PD heating rate
       tt = sig_pd*isrfuv(jtresh:jfreqmax)*(hnu-ipdt+emin)/hnu
       hp2 = XINTEG2( 1, nfrq, nfrq, freq_qabs(jtresh:jfreqmax), tt )
       ! PD current
       tt = tt/(hnu-ipdt+emin)/everg
       jp2 = XINTEG2( 1, nfrq, nfrq, freq_qabs(jtresh:jfreqmax), tt )

    ELSE   ! no photons of energies above PD threshhold

       jp2 = 0.0_dp
       hp2 = 0.0_dp
       hsp2 = 0.0_dp

    ENDIF
    
    jp = jp1 + jp2
    hp = hp1 + hp2
    hsp = hsp1 + hsp2

  END SUBROUTINE PE_CURRENT

!----------------------------------------------------------------------------

END MODULE MZDIST
