MODULE MCOMPUTE

  USE CONSTANTS
  IMPLICIT NONE

  PRIVATE
  PUBLIC :: COMPUTE

CONTAINS

! ------------------------------------------------------------------------------------

SUBROUTINE COMPUTE

  ! global variables modules

  USE CONSTANTS
  USE UTILITY
  USE MGET_QEXT
  USE MGET_TDIST
  USE MZDIST
  USE MSPIN
  USE MDTLS

  ! local variables

  IMPLICIT NONE

  TYPE (FICH)                 :: ftemp,fzdist,fsed_a,fpsed_a,fspd_a
  CHARACTER (LEN=max_len)     :: filename_tmp,c_tmp

  ! loop indices
  INTEGER                     :: i             ! grain type index
  INTEGER                     :: j             ! grain size index
  INTEGER                     :: kt

  REAL (KIND=dp)              :: tt
  REAL (KIND=dp)              :: enerabs(nsize_max)  ! power absorbed by grain
  REAL (KIND=dp)              :: enerem(nsize_max)   ! power emitted by grain
  REAL (KIND=dp)              :: tempmoy(nsize_max)  ! grain mean temperature 
  REAL (KIND=dp)              :: tempequi(nsize_max) ! grain equilibrium temperature 
  REAL (KIND=dp)              :: tempmax(nsize_max)  ! grain max temperature 
  REAL (KIND=dp)              :: t_rot(nsize_max)    ! rotational temperature

  REAL (KIND=dp)              :: temp(ndist)   ! temperature grid
  REAL (KIND=dp)              :: t2(ndist)     ! LOG(T)
  REAL (KIND=dp)              :: hcap(ndist)   ! heat capacity
  REAL (KIND=dp)              :: dpt(ndist)    ! dP/dln T
  REAL (KIND=dp)              :: uint(ndist)   ! internal energy
  REAL (KIND=dp)              :: dpu(ndist)    ! dP/dU

  ! for dust IR (vibrational) SED
  REAL (KIND=dp), ALLOCATABLE :: nuflux_j(:)   ! flux*frequency for 1 type and 1 size
  REAL (KIND=dp), ALLOCATABLE :: nuflux(:,:)   ! flux*frequency for 1 type and all sizes

  ! for dust polarized emission
  REAL (KIND=dp), ALLOCATABLE :: nufluxp_j(:)  ! flux*frequency for 1 type and 1 size
  REAL (KIND=dp), ALLOCATABLE :: nufluxp(:,:)  ! flux*frequency for 1 type and all sizes

  ! for charge and mix
  INTEGER                     :: niter,nitermax
  REAL (KIND=dp)              :: sd_z
  REAL (KIND=dp), ALLOCATABLE :: fz0(:)
  REAL (KIND=dp)              :: nx,tmp,norm,eps,amn,f0,fm,zneg,cx,atm,ft0,ftm,ztn  ! fractions of neutrals, charged and mean Z<0
  REAL (KIND=DP)              :: zmean(nsize_max),ztot(ntype),z1(ntype)
  
  ! for spinning dust SED
  REAL (KIND=dp), ALLOCATABLE :: nuf(:)
  REAL (KIND=dp), ALLOCATABLE :: spnuflux_j(:)  ! flux*frequency for 1 type and 1 size
  REAL (KIND=dp), ALLOCATABLE :: spnuflux(:,:)  ! flux*frequency for 1 type and all sizes

  ! for integration overs sizes
  INTEGER                     :: ns_i
  REAL (KIND=dp)              :: xx(nsize_max), yy(nsize_max)       ! Integrand

  !-------------------------------------------------------------------------
  ftemp%nom   = 'TEMP';         ftemp%unit   = 35
  fzdist%nom  = 'ZDIST';        fzdist%unit  = 36
  fsed_a%nom  = 'SED_A';        fsed_a%unit  = 34
  fpsed_a%nom = 'SED_POL_A';    fpsed_a%unit = 32
  fspd_a%nom  = 'SPIN_A';       fspd_a%unit  = 33

  IF (LEN(TRIM(rname)) > 0) THEN
     c_tmp=ftemp%nom;ftemp%nom=TRIMCAT(c_tmp,'_'//TRIM(rname)//'.RES')
     c_tmp=fzdist%nom;fzdist%nom=TRIMCAT(c_tmp,'_'//TRIM(rname)//'.RES')
     c_tmp=fsed_a%nom;fsed_a%nom=TRIMCAT(c_tmp,'_'//TRIM(rname)//'.RES')
     c_tmp=fpsed_a%nom;fpsed_a%nom=TRIMCAT(c_tmp,'_'//TRIM(rname)//'.RES')
     c_tmp=fspd_a%nom;fspd_a%nom=TRIMCAT(c_tmp,'_'//TRIM(rname)//'.RES')
  ELSE
     ftemp%nom=TRIMCAT(ftemp%nom,'.RES')
     fzdist%nom=TRIMCAT(fzdist%nom,'.RES')
     fsed_a%nom=TRIMCAT(fsed_a%nom,'.RES')
     fpsed_a%nom=TRIMCAT(fpsed_a%nom,'.RES')
     fspd_a%nom=TRIMCAT(fspd_a%nom,'.RES')
  ENDIF
  
  ! allocate IR SED arrays 
  ALLOCATE (nuflux_j(n_qabs))
  ALLOCATE (nuflux(nsize_max,n_qabs))

  ! allocate polarized SED arrays
  IF (n_pol > 0 .and. n_lin > 0) THEN 
     ALLOCATE (nufluxp_j(n_qabs))
     ALLOCATE (nufluxp(nsize_max,n_qabs))
  ENDIF

  ! allocate and initialize charge distribution arrays
  IF (n_chrg > 0) THEN 
     ALLOCATE (jpe(nsize_max),jgas(nsize_max),hpe(nsize_max),hspe(nsize_max),cgas(nsize_max))
  ENDIF
  f0 = 1.0_dp
  ztot = 0.0_dp; z1 = 0.0_dp

  ! allocate spinning arrays
  IF (n_spin > 0) THEN 
     ALLOCATE (spnuflux_j(n_qabs))
     ALLOCATE (spnuflux(nsize_max, n_qabs))
  ENDIF

  IF (n_ftemp > 0) THEN
     filename_tmp = TRIMCAT(data_path,dir_res)
     filename_tmp = trimcat(filename_tmp,ftemp%nom)
     OPEN (UNIT=ftemp%unit, FILE = filename_tmp, STATUS = 'unknown')
     WRITE (UNIT=ftemp%unit, FMT='(A40)')'# DUSTEM: grain temperature distribution'
     WRITE (UNIT=ftemp%unit, FMT='(A1)') '#'
     WRITE (UNIT=ftemp%unit, FMT='(A119)') &
     & '# TYPE                   size(cm)          Teq(K)              Tmoy(K)            Tmax(K)        Trot(K)          ndist'
     WRITE (UNIT=ftemp%unit, FMT='(A1)') '#'
     IF (n_ft > 0) THEN 
        WRITE (UNIT=ftemp%unit, FMT='(A87)') &
             & '#       T(K)             dP/dlnT           C(T)(erg/K)         U(erg)             dP/dU'
        WRITE (UNIT=ftemp%unit, FMT='(A1)') '#'
     ENDIF
  ENDIF

  IF ((n_zdist > 0) .AND. (n_chrg > 0)) THEN
     filename_tmp = TRIMCAT(data_path,dir_res)
     filename_tmp = trimcat(filename_tmp,fzdist%nom)
     OPEN (UNIT=fzdist%unit, FILE = filename_tmp, STATUS = 'unknown')
     WRITE (UNIT=fzdist%unit, FMT='(A35)')'# DUSTEM: grain charge distribution'
     WRITE (UNIT=fzdist%unit, FMT='(A1)') '#'
     WRITE (UNIT=fzdist%unit, FMT='(A80,A108)') &
     & '# TYPE                  nzb    size(cm)        Zmin           Zmax           Zeq', &
     & '         Zmean          sd_Z           Jpe(s-1)      Jgas(s-1)      Pabs(erg/s)     Cgas/Pabs      Hpe/Pabs'
     WRITE (UNIT=fzdist%unit, FMT='(A1)') '#'
     IF (n_fz > 0) THEN 
        WRITE (UNIT=fzdist%unit, FMT='(A31)') '#         Z                f(Z)'
        WRITE (UNIT=fzdist%unit, FMT='(A1)') '#'
     ENDIF
  ELSE IF ((n_zdist > 0) .AND. (n_chrg == 0)) THEN
     WRITE (*,*) '(W) ZDIST set without CHRG (no charge computed)'
  ENDIF

  IF (n_res_a > 0) THEN
     filename_tmp = TRIMCAT(data_path, dir_res)
     filename_tmp = trimcat(filename_tmp,fsed_a%nom)
     OPEN (UNIT = fsed_a%unit, FILE = filename_tmp, STATUS = 'unknown')
     WRITE (UNIT = fsed_a%unit, FMT='(A58)') '# DUSTEM: grain SED per size 4*pi*nu*I_nu/Ng (erg/s/grain)'
     WRITE (UNIT = fsed_a%unit, FMT='(A1)')  '#'
     WRITE (UNIT = fsed_a%unit, FMT='(A33)') '# TYPE  nr of sizes   nr of waves'
     WRITE (UNIT = fsed_a%unit, FMT='(A28)') '# size(1)...size(nsize) (cm)'
     WRITE (UNIT = fsed_a%unit, FMT='(A38)') '# lambda(microns) SED(1)....SED(nsize)'
     WRITE (UNIT = fsed_a%unit, FMT='(A1)')  '#'

     IF (n_pol > 0 .and. n_lin > 0) THEN 
        filename_tmp = TRIMCAT(data_path, dir_res)
        filename_tmp = trimcat(filename_tmp,fpsed_a%nom)
        OPEN (UNIT = fpsed_a%unit, FILE = filename_tmp, STATUS = 'unknown')
        WRITE (UNIT = fpsed_a%unit, FMT='(A68)') '# DUSTEM: grain polarized SED per size 4*pi*nu*I_nu/Ng (erg/s/grain)'
        WRITE (UNIT = fpsed_a%unit, FMT='(A1)')  '#'
        WRITE (UNIT = fpsed_a%unit, FMT='(A33)') '# TYPE  nr of sizes   nr of waves'
        WRITE (UNIT = fpsed_a%unit, FMT='(A28)') '# size(1)...size(nsize) (cm)'
        WRITE (UNIT = fpsed_a%unit, FMT='(A38)') '# lambda(microns) SED(1)....SED(nsize)'
        WRITE (UNIT = fpsed_a%unit, FMT='(A1)')  '#'
     ENDIF
  ENDIF

  IF (n_spin > 0) THEN
     filename_tmp = TRIMCAT(data_path, dir_res)
     filename_tmp = trimcat(filename_tmp,fspd_a%nom)
     OPEN (UNIT = fspd_a%unit, FILE = filename_tmp, STATUS = 'unknown')
     WRITE (UNIT = fspd_a%unit, FMT='(A67)') '# DUSTEM: grain spinning SED per type 4*pi*nu*I_nu/Ng (erg/s/grain)'
     WRITE (UNIT = fspd_a%unit, FMT='(A1)')  '#'
     WRITE (UNIT = fspd_a%unit, FMT='(A33)') '# TYPE  nr of types   nr of waves'
     WRITE (UNIT = fspd_a%unit, FMT='(A28)') '# size(1)...size(nsize) (cm)'
     WRITE (UNIT = fspd_a%unit, FMT='(A38)') '# lambda(microns) SED(1)....SED(ntype)'
     WRITE (UNIT = fspd_a%unit, FMT='(A1)')  '#'
  ENDIF

  IF (n_chrg > 0 .OR. n_spin > 0) THEN 
     IF (n_quiet == 0) THEN
        WRITE(*,*)
        WRITE(*,FMT='(A14)')'Gas quantities'
        WRITE(*,FMT='(A6,1PE9.2,2(A10,1PE9.2))') 'Tgas =',t_gas,' nH =',hden, ' n(H2) =',h2den
        if (iden(3)>=0.0_dp) then
           c_tmp = TRIM('ionization (CR and metals only)     : x =')
        else
           c_tmp = TRIM('ionization (CR, metals and C+)      : x =')
        endif
        WRITE (*, FMT='(A41,1PE9.2,2(A6,1PE9.2),2(A10,1PE9.2))') &
             & c_tmp,eden/hden,'  xp =',iden(2)/hden,'  xm =',xm, &
             & ' /*\ xdc =',xdc,' xcr^0.5 =',SQRT(xcr)
     ENDIF

  ! charge exchange
     IF ((iden(3) < 0.0_dp) .AND. (MAXVAL(cex)>0.0_dp)) THEN
        iden(3)=ABS(iden(3)); eps = 1.0_dp; niter=0; nitermax=50
        DO WHILE (eps>1e-2_dp)
           cx=0.0_dp; nx=0.0_dp; ft0=0.0_dp; ftm=0.0_dp; ztn=0.0_dp; atm=0.0_dp
           DO i=1,ntype
              IF (cex(i) > 0.0_dp) THEN
                 norm = 0.0_dp; nx = nx+1.0_dp; cx = cx+cex(i)
                 f0=0.0_dp; fm=0.0_dp; zneg=0.0_dp; amn=0.0_dp
                 DO j=1,nsize(i)           
                    CALL ZDIST(i, j, size_ava(i,j), zmean(j), sd_z, jpe(j), hpe(j), hspe(j), jgas(j), cgas(j))
                    ! get charge fractions
                    ALLOCATE (fz0(nzb))
                    fz0 = fz
                    WHERE(zb/=0.0_dp) fz0=0.0_dp  ! neutrals
                    f0 = f0 + SUM(fz0) * ava(i,j)/size_ava(i,j)**3  
                    fz0 = fz
                    WHERE(zb>=0.0_dp) fz0=0.0_dp  ! negative only          
                    tmp = SUM(fz0)
                    fm = fm + tmp * ava(i,j)/size_ava(i,j)**3
                    if (tmp>0_dp) zneg = zneg + (SUM(fz0*zb)/tmp) * ava(i,j)/size_ava(i,j)**3
                    amn = amn + ava(i,j)/size_ava(i,j)**2
                    norm = norm +  ava(i,j)/size_ava(i,j)**3
                    DEALLOCATE(zb,fz,fz0)
                 ENDDO   ! end size loop
                 ft0=ft0+f0*xmh*mprop(i)/masstot(i); ftm=ftm+fm*xmh*mprop(i)/masstot(i)
                 ztn=ztn+zneg/norm; atm=atm+amn/norm*1e7_dp
              ENDIF
           ENDDO      ! end type loop
           ft0=ft0; ftm=ftm; ztn=ztn/nx; atm=atm/nx; cx=cx/nx
           tmp = hden * XSOLVE(atm,ft0,ftm,ztn,cx)
           eps = ABS(2.0_dp*(eden - tmp)/(eden+tmp))
           eden = tmp  ! update ionization values
           eden = MAX(eden, xdc*hden) ! DC case
           xm = eden/hden - 1.0_dp/(1.0_dp + eden/hden/xcr)
           xm = MIN(xm,iden(3)/hden)  ! make sure that n(C+) =< n(C)=ABS(iden(3))
           iden(1) = eden
           iden(2) = eden - xm*hden
           iden(3) = xm*hden
           niter = niter + 1
           if (niter>nitermax) then
              print*,' (W) DM_COMPUTE: charge exchange not converged after ',INT(nitermax),' iterations'
              print*,'                 exiting with current electron density eden.'
              eps = 1e-3_dp
           endif
        ENDDO  ! end while
        IF (n_quiet == 0) THEN
           c_tmp = TRIM('ionization (CR, metals, C+ and dust): x =')
           WRITE (*, FMT='(A41,1PE9.2,2(A6,1PE9.2))') &
                & c_tmp,eden/hden,'  xp =',iden(2)/hden,'  xm =',iden(3)/hden
        ENDIF
     ENDIF
  ENDIF
  
  ! compute and write
  c_tmp = ''
  n_zm = 0
  
  DO i=1,ntype                                     ! loop on grain types

     t_rot(:) = 0.0_dp
     ns_i     = nsize(i)
     n_beta   = 0
     n_dtls   = 0
     IF (INDEX(t_opt(i),'BETA') > 0)   n_beta   = 1 
     IF (INDEX(t_opt(i),'DTLS') > 0)   n_dtls   = 1 

     DO j=1,nsize(i)                               ! loop on sizes

        CALL GET_TDIST( &
             & i,             &                    ! (I): index of grain TYPE
             & j,             &                    ! (I): index of grain size
             & size_ava(i,j), &                    ! (I): grain size
             & enerabs(j),    &                    ! (O): power absorbed by grain
             & uint,          &                    ! (O): internal energy grid
             & dpu,           &                    ! (O): dP/dU
             & temp,          &                    ! (O): grain temperature T
             & dpt,           &                    ! (O): dP/dln T
             & hcap,          &                    ! (O): grain heat capacity vs temp
             & tempmoy(j) ,   &                    ! (O): average temperature 
             & tempequi(j),   &                    ! (O): equilibrium temperature 
             & tempmax(j) )                        ! (O): max temperature 

        t2(:) = LOG(temp(:))

        CALL GET_ASED (i,j,size_ava(i,j),ndist,temp(1:ndist),t2(1:ndist),dpt(1:ndist),nuflux_j(:),enerem(j))
        nuflux(j,:) = nuflux_j(:)

        IF (n_pol > 0 .and. n_lin > 0 .and. INDEX(t_opt(i),'POL') > 0) THEN
           CALL GET_ASED_POL (i,j,size_ava(i,j),ndist,temp(1:ndist),t2(1:ndist),dpt(1:ndist),nufluxp_j(:))
           nufluxp(j,:) = nufluxp_j(:)
        ENDIF

        IF ((INDEX(t_opt(i),'CHRG') > 0 ) .OR. (INDEX(t_opt(i),'SPIN') > 0 )) THEN
           CALL ZDIST(i, j, size_ava(i,j), zmean(j), sd_z, jpe(j), hpe(j), hspe(j), jgas(j), cgas(j))

           ! get neutrals
           ALLOCATE (fz0(nzb))
           fz0 = fz
           WHERE(zb/=0.0_dp) fz0=0.0_dp  
           f0 = SUM(fz0)  ! neutral
           IF (INDEX(t_opt(i),'ZM') > 0) THEN
              if (n_zm == 0) then                    ! zm: 1st pass
                 f_mix(i,j) = f0                     ! neutrals
              ELSE if (n_zm == 1) then               ! zm: 2nd pass
                 f_mix(i,j) = 1.0_dp - f_mix(i-1,j)  ! all charged species
              ENDIF
           ENDIF
          
           IF (n_zdist > 0) THEN 
              WRITE (UNIT = fzdist%unit, FMT='(A2,A20,1x,i4,1x,11(1PE12.4,3X))') '# ',gtype(i),nzb,size_ava(i,j),zmin,zmax,zeq, &
                   & zmean(j),sd_z,jpe(j),jgas(j),enerabs(j),cgas(j)/enerabs(j),hpe(j)/enerabs(j)
              IF (n_fz > 0) THEN
                 DO kt = 1, nzb
                    WRITE (UNIT = fzdist%unit, FMT='(1P,2(E18.10E3,1X))') zb(kt),fz(kt)
                 ENDDO
              ENDIF
           ENDIF
          
        ENDIF

        IF (INDEX(t_opt(i),'SPIN') > 0 ) THEN
           ALLOCATE (nuf(n_qabs))
           DO kt=1,n_qabs  ! grain Inu (erg/s/Hz/sr)   !!! NB reverse order for SPIN (frequency order)
              nuf(kt) = nuflux_j(n_qabs-kt+1) * enerabs(j)/enerem(j)  / 4.0_dp/xpi / freq_qabs(kt)
           ENDDO
           CALL SPIN(i, j, size_ava(i,j), tempequi(j), nuf, spnuflux_j, t_rot(j))
           DEALLOCATE (nuf)
           spnuflux(j,:) = spnuflux_j(:)
           nuflux(j,:) = nuflux(j,:) + spnuflux(j,:)
        ENDIF

        IF (n_ftemp > 0) THEN
           WRITE (UNIT = ftemp%unit, FMT='(A2,A20,1x,5(1PE12.4,6X),i4)') '# ',gtype(i),size_ava(i,j),tempequi(j), &
                & tempmoy(j),tempmax(j),t_rot(j),ndist
           IF (n_ft > 0) THEN 
              DO kt = 1, ndist
                 WRITE (UNIT = ftemp%unit, FMT='(1P,5(E18.10E3,1X))') temp(kt), dpt(kt), hcap(kt), uint(kt), dpu(kt)
              ENDDO
           ENDIF           
        ENDIF

        IF ((INDEX(t_opt(i),'CHRG')>0) .OR. (INDEX(t_opt(i),'SPIN')>0)) THEN 
           DEALLOCATE(zb,fz,fz0)
        ENDIF

     ENDDO    ! end of size loop (j)

     xx(1:ns_i) = si_ava_l(i,1:ns_i)   ! size log scale per type

     !!!!! apply mixing (default f_mix is 1) !!!!!
     ava(i,1:ns_i) = ava(i,1:ns_i)*f_mix(i,1:ns_i)     

     ! get SED in erg/s/H (4*pi*nu*inu/NH)
     DO kt=1,n_qabs
       yy(1:ns_i) = ( nuflux(1:ns_i,kt)/ size_ava(i,1:ns_i)**3 ) * (enerabs(1:ns_i) / enerem(1:ns_i)) * ava(i,1:ns_i)
       nuinuem(i,kt) = XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i))
     ENDDO
     nuinuem(i,:) = nuinuem(i,:) * mprop(i) * xmp / masstot(i)
     nuinuemtot = nuinuemtot + nuinuem(i,:)

     IF (n_res_a > 0) THEN
        WRITE (UNIT=fsed_a%unit, FMT='(A2,a20,1x,i3,1X,I4)') '# ', gtype(i), nsize(i), n_qabs
        WRITE (UNIT=fsed_a%unit, FMT='(100(1PE12.4,1X))')  (size_ava(i,j), j=1,nsize(i))
        DO kt=1,n_qabs
           WRITE (UNIT=fsed_a%unit, FMT='(101(1PE14.6E3,1X))') lamb_qabs(kt)*1.e4_dp, &
                (nuflux(j,kt)*enerabs(j)/enerem(j), j=1,nsize(i))
        ENDDO
     ENDIF

     IF (INDEX(t_opt(i),'CHRG') > 0) THEN ! get heating and cooling per type
        yy(1:ns_i) = (ava(i,1:ns_i) / size_ava(i,1:ns_i)**3)
        tt = XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i))
        ztot(i) = XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i)*zmean(1:ns_i)) * mprop(i)*xmp/masstot(i)
        z1(i) = ztot(i) / (XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i)) * mprop(i)*xmp/masstot(i))
        hpt(i) = XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i)*hpe(1:ns_i)) * mprop(i)*xmp/masstot(i)
        hspt(i) = XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i)*hspe(1:ns_i)) * mprop(i)*xmp/masstot(i)
        cgt(i) = XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i)*cgas(1:ns_i)) * mprop(i)*xmp/masstot(i)
        e_pt(i) = XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i)*hpe(1:ns_i)/enerabs(1:ns_i)) / tt
        e_gt(i) = XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i)*cgas(1:ns_i)/enerabs(1:ns_i)) / tt
             
     ENDIF
     
     IF (INDEX(t_opt(i),'SPIN') > 0 ) THEN
        DO kt=1,n_qabs
           yy(1:ns_i) = ( spnuflux(1:ns_i,kt)/ size_ava(i,1:ns_i)**3 ) &
                * (enerabs(1:ns_i) / enerem(1:ns_i)) * ava(i,1:ns_i)
           spnuinuem(i,kt) = XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i))
        ENDDO
        spnuinuem(i,:) = spnuinuem(i,:) * mprop(i) * xmp / masstot(i)
        spnuinuemtot = spnuinuemtot + spnuinuem(i,:)
        IF (n_res_a > 0) THEN
           WRITE(UNIT = fspd_a%unit, FMT='(A2,a20,1x,i3,1X,I4)') '# ', gtype(i), nsize(i), n_qabs
           WRITE(UNIT = fspd_a%unit, FMT='(100(1PE12.4,1X))')  (size_ava(i,j), j=1,nsize(i))
           DO kt=1,n_qabs
              WRITE(UNIT = fspd_a%unit, FMT='(101(1PE14.6E3,1X))') lamb_qabs(kt)*1.e4_dp, (spnuflux(j,kt), j=1,nsize(i))
           ENDDO
        ENDIF        
     ENDIF

     IF (n_pol > 0 .and. n_lin > 0 .and. INDEX(t_opt(i),'POL') > 0) THEN 
        DO kt=1,n_qabs
           yy(1:ns_i) = ( nufluxp(1:ns_i,kt)/ size_ava(i,1:ns_i)**3 ) &
                * (enerabs(1:ns_i) / enerem(1:ns_i)) * ava(i,1:ns_i)
           nuinuemp(i,kt) = XINTEG2 (1, ns_i, ns_i, xx(1:ns_i), yy(1:ns_i))
        ENDDO
        nuinuemp(i,:) = nuinuemp(i,:) * mprop(i) * xmp / masstot(i)
        nuinuemptot = nuinuemptot + nuinuemp(i,:)
        IF (n_res_a > 0) THEN
           WRITE(UNIT = fpsed_a%unit, FMT='(A2,a20,1x,i3,1X,I4)') '# ', gtype(i), nsize(i), n_qabs
           WRITE(UNIT = fpsed_a%unit, FMT='(100(1PE12.4,1X))')  (size_ava(i,j), j=1,nsize(i))
           DO kt=1,n_qabs
              WRITE(UNIT = fpsed_a%unit, FMT='(101(1PE14.6E3,1X))') lamb_qabs(kt)*1.e4_dp, (nufluxp(j,kt), j=1,nsize(i))
           ENDDO
        ENDIF
     ENDIF

     ! update n_zm for CHRG pass
     IF ( ((INDEX(t_opt(i),'CHRG') > 0) .AND. (INDEX(t_opt(i),'ZM') > 0 ) .AND. (n_zm < 1)) &                ! chrg-zm or
          & .OR. ((INDEX(t_opt(i),'SPIN') > 0) .AND. (INDEX(t_opt(i),'ZM') > 0 ) .AND. (n_zm < 1)) ) THEN    ! spin-zm 1st pass
        n_zm = n_zm + 1   ! increment to 2d pass
     ELSE 
        n_zm = 0
     ENDIF
     
     c_tmp = TRIM(c_tmp)//' '//TRIM(gtype(i))
     
  ENDDO  ! end type loop (i)

  c_tmp = TRIM(c_tmp)//' total'
  
  ! deallocate arrays
  DEALLOCATE(nuflux_j)
  DEALLOCATE(nuflux)
  IF (n_pol > 0 .and. n_lin > 0) THEN 
     DEALLOCATE (nufluxp_j)
     DEALLOCATE (nufluxp)
  ENDIF
  IF (n_spin > 0) THEN 
     DEALLOCATE(spnuflux_j)
     DEALLOCATE(spnuflux)
  ENDIF

  ! close files
  IF ((n_zdist > 0) .AND. (n_chrg > 0)) THEN  ! global quantities and ionized fractions
     WRITE (UNIT=fzdist%unit, FMT='(A1)') '#'
     WRITE (UNIT=fzdist%unit, FMT='(A1,A50)')'#',c_tmp
     WRITE (UNIT=fzdist%unit, FMT='(A20)')'# Grain charge per H'
     tt = SUM( (/ (ztot(i), i=1,ntype) /) )
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') (ztot(i), i=1,ntype), tt
     WRITE (UNIT=fzdist%unit, FMT='(A23)')'# Mean charge per grain'
     tt = SUM( (/ (z1(i), i=1,ntype) /) )
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') (z1(i), i=1,ntype), tt
     WRITE (UNIT=fzdist%unit, FMT='(A24)') '# Gas heating rate per H'
     tt = SUM( (/ (hpt(i), i=1,ntype) /) )
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') (hpt(i), i=1,ntype), tt
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') (hpt(i)/tt, i=1,ntype)
     WRITE (UNIT=fzdist%unit, FMT='(A21)') '# Heating efficiency'
     tt = SUM( (/ (e_pt(i), i=1,ntype) /) )
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') (e_pt(i), i=1,ntype), tt
     WRITE (UNIT=fzdist%unit, FMT='(A24)') '# Gas cooling rate per H'
     tt = SUM( (/ (cgt(i), i=1,ntype) /) )
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') (cgt(i), i=1,ntype), tt
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') (cgt(i)/tt, i=1,ntype)
     WRITE (UNIT=fzdist%unit, FMT='(A21)') '# Cooling efficiency'
     tt = SUM( (/ (e_gt(i), i=1,ntype) /) )
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') (e_gt(i), i=1,ntype), tt
     WRITE (UNIT=fzdist%unit, FMT='(A28)') '# Spin excitation rate per H'
     tt = SUM( (/ (hspt(i), i=1,ntype) /) )
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') (hspt(i), i=1,ntype), tt
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') (hspt(i)/tt, i=1,ntype)
     WRITE (UNIT=fzdist%unit, FMT='(A1)') '#'
     WRITE (UNIT=fzdist%unit, FMT='(A66)') '# Gas Summary: T, nH, n(H2), CR, G0 / x, x_p, x_M, sqrt(xcr), x_DC'
     WRITE (UNIT=fzdist%unit, FMT='(1P,10E13.4E3)') t_gas,hden,h2den,cr_rate,g0 
     WRITE (UNIT=fzdist%unit, FMT='(1P,20E13.4E3)') eden/hden,iden(2)/hden,xm,SQRT(xcr),xdc
     CLOSE (UNIT = fzdist%unit)
  ENDIF
  IF (n_ftemp > 0) CLOSE (UNIT = ftemp%unit)
  IF (n_res_a > 0) THEN 
     CLOSE (UNIT = fsed_a%unit)
     IF (n_pol > 0 .and. n_lin > 0) CLOSE (UNIT = fpsed_a%unit)
  ENDIF

END SUBROUTINE COMPUTE

! -------------------------------------------------------------------------------

SUBROUTINE GET_ASED (nt, ns, a, ntsed, t, ti, p, xnufnu_los, energiemise)

! computes the SED (erg/s) emitted by a grain of given size and with  
! temperature distribution p (usually dp/dlnT). 
! Temperature grid for integration is ti (usually LOG(T))
! uses lambda*B_lambda(T)

  USE CONSTANTS
  USE UTILITY
  USE MGET_QEXT
  USE MDTLS

  IMPLICIT NONE

  INTEGER, INTENT (IN)         :: nt               ! index of grain type
  INTEGER, INTENT (IN)         :: ns               ! index of grain size
  INTEGER, INTENT (IN)         :: ntsed            ! nr of T-values
  REAL (KIND=dp), INTENT (IN)  :: a                ! grain size
  REAL (KIND=dp), INTENT (IN)  :: t(ntsed)
  REAL (KIND=dp), INTENT (IN)  :: ti(ntsed)
  REAL (KIND=dp), INTENT (IN)  :: p(ntsed)
  ! Anisotropic emission : energy emitted toward the observer
  REAL (KIND=dp), INTENT (OUT) :: xnufnu_los(n_qabs)
  ! Isotropic emission : energy emitted in all directions Only useful to calculate energiemise
  REAL (KIND=dp)               :: xnufnu(n_qabs)
  REAL (KIND=dp), INTENT (OUT) :: energiemise

  INTEGER        :: l, ktemp
  REAL (KIND=dp) :: al1
  REAL (KIND=dp) :: fnusom
  REAL (KIND=dp) :: fnut(ntsed), xx(ntsed)
  REAL (KIND=dp) :: yy(n_qabs)
  REAL (KIND=dp) :: coef
  REAL (KIND=dp) :: sig(ntsed)

  coef = xpi * a**2 * cte1

  ! main loop on wavelengths
  energiemise = 0.0_dp
  IF (n_beta == 0 .AND. n_dtls == 0) THEN
     DO l=1,n_qabs
        al1 = lamb_qabs(l)
        ! temperature loop
        xx(:) = hcsurk / (al1 * t(:))
        DO ktemp=1,ntsed
           fnut(ktemp) = p(ktemp) * F_BB (xx(ktemp))
        ENDDO
        fnusom = XINTEG2 (1, ntsed, ntsed, ti, fnut)
        ! qauv is backward...
        xnufnu(l) = coef * fnusom * qi_abs(nt,ns,n_qabs-l+1) / al1**4    
        IF (n_anis .eq. 0 .or. INDEX(t_opt(nt),'POL') == 0) THEN
            ! We ignore alignment for total emission
            xnufnu_los(l) = xnufnu(l)
        ELSE 
            ! We take alignment into account for total emission
            ! Contribution of not aligned grains (1-f) + that of aligned grains (f)
           xnufnu_los(l) = coef * fnusom / al1**4 &
                * ( (1. - f_pol(nt,ns)) * qi_abs(nt,ns,n_qabs-l+1) &
                + f_pol(nt,ns) * (q1i_abs(nt,ns,n_qabs-l+1)+q2i_abs(nt,ns,n_qabs-l+1))/2 )
        ENDIF
     ENDDO
  ELSE IF (n_beta == 1 .AND. n_dtls == 0) THEN    ! apply BETA(T)
     DO l=1,n_qabs
        al1 = lamb_qabs(l)
        ! temperature loop
        xx(:) = hcsurk / (al1 * t(:))
        DO ktemp=1,ntsed
           fnut(ktemp) = p(ktemp) * F_BB (xx(ktemp)) * &
           ! faster
           & EXP( DBETA(t(ktemp),nt)*f_beta(nt,l)*LOG(ltresh(nt)/al1) )
        ENDDO
        fnusom = XINTEG2 (1, ntsed, ntsed, ti, fnut)
        ! qauv is backward...
        xnufnu(l) = coef * fnusom * qi_abs(nt,ns,n_qabs-l+1) / al1**4 
     ENDDO
     xnufnu_los = xnufnu
  ELSE IF (n_beta == 0 .AND. n_dtls == 1) THEN  ! apply DCD-TLS
     DO l=1,n_qabs
        al1 = lamb_qabs(l)
        ! temperature loop
        xx(:) = hcsurk / (al1 * t(:))
        IF (al1*1e4_dp < ldtresh(nt)) THEN 
           DO ktemp=1,ntsed
              fnut(ktemp) = p(ktemp) * F_BB (xx(ktemp)) * qi_abs(nt,ns,n_qabs-l+1)
           ENDDO
        ELSE 
           CALL DTLS (nt, ntsed, ns, n_qabs-l+1, a, t, sig)
           DO ktemp=1,ntsed
              fnut(ktemp) = p(ktemp) * F_BB (xx(ktemp)) * sig(ktemp)
           ENDDO
        ENDIF
        fnusom = XINTEG2 (1, ntsed, ntsed, ti, fnut)
        xnufnu(l) = coef * fnusom / al1**4 
     ENDDO
     xnufnu_los = xnufnu     
  ENDIF

  yy(:) = xnufnu(:) / lamb_qabs(:)
  l = 1
  energiemise = XINTEG2 (l, n_qabs, n_qabs, lamb_qabs, yy)

END SUBROUTINE GET_ASED

! -------------------------------------------------------------------------------

SUBROUTINE GET_ASED_POL (nt, ns, a, ntsed, t, ti, p, xnufnu)

! computes the SED (erg/s) emitted by a grain of given size and with  
! temperature distribution p (usually dp/dlnT). 
! Temperature grid for integration is ti (usually LOG(T))
! uses lambda*B_lambda(T)

  USE CONSTANTS
  USE UTILITY
  USE MGET_QEXT

  IMPLICIT NONE

  INTEGER, INTENT (IN)         :: nt               ! index of grain type
  INTEGER, INTENT (IN)         :: ns               ! index of grain size
  INTEGER, INTENT (IN)         :: ntsed            ! nr of T-values
  REAL (KIND=dp), INTENT (IN)  :: a                ! grain size
  REAL (KIND=dp), INTENT (IN)  :: t(ntsed)
  REAL (KIND=dp), INTENT (IN)  :: ti(ntsed)
  REAL (KIND=dp), INTENT (IN)  :: p(ntsed)
  REAL (KIND=dp), INTENT (OUT) :: xnufnu(n_qabs)

  INTEGER        :: l, ktemp
  REAL (KIND=dp) :: al1
  REAL (KIND=dp) :: fnusom
  REAL (KIND=dp) :: fnut(ntsed), xx(ntsed)
  REAL (KIND=dp) :: coef

  coef = xpi * a**2 * cte1

  ! main loop on wavelengths
  IF (n_beta == 0) THEN
     DO l=1,n_qabs
        al1 = lamb_qabs(l)
        ! temperature loop
        xx(:) = hcsurk / (al1 * t(:))
        DO ktemp=1,ntsed
           fnut(ktemp) = p(ktemp) * F_BB (xx(ktemp))
        ENDDO
        fnusom = XINTEG2 (1, ntsed, ntsed, ti, fnut)
        ! qauv is backward...
        xnufnu(l) = coef * fnusom * (q2i_abs(nt,ns,n_qabs-l+1)-q1i_abs(nt,ns,n_qabs-l+1))/2 / &
             & al1**4    
     ENDDO
  ELSE                  ! apply BETA(T)
     DO l=1,n_qabs
        al1 = lamb_qabs(l)
        ! temperature loop
        xx(:) = hcsurk / (al1 * t(:))
        DO ktemp=1,ntsed
           fnut(ktemp) = p(ktemp) * F_BB (xx(ktemp)) * &
           ! faster
           & EXP( DBETA(t(ktemp),nt)*f_beta(nt,l)*LOG(ltresh(nt)/al1) )
        ENDDO
        fnusom = XINTEG2 (1, ntsed, ntsed, ti, fnut)
        ! qauv is backward...
        xnufnu(l) = coef * fnusom * (q2i_abs(nt,ns,n_qabs-l+1)-q1i_abs(nt,ns,n_qabs-l+1))/2 / &
             & al1**4 
     ENDDO
  ENDIF

  xnufnu(:) = xnufnu(:) * f_pol(nt,ns) 

END SUBROUTINE GET_ASED_POL

END MODULE MCOMPUTE
