
from numpy import *

def serkowski( x=[], ka=0, xmax=0, pmax=0 ):

    if size(x) == 0:
        print('----------------------------------------------------------------------')
        print('def serkowski( x=[], ka=ka, xmax=xmax, pmax=pmax)')
        print('----------------------------------------------------------------------')
        print(' computes the Serkowski law (Draine & Fraisse 2009)')
        print()
        print(' X    (I): array(n_qabs) inverse wavenumber in 1/microns')
        print(' KA   (I): K factor')
        print(' XMAX (I): x-position of max')
        print(' PMAX (I): max polar. fraction')

    if ka == 0: ka = 0.92  # Draine & Fraisse (2009)
    if xmax == 0: xmax = 1.82
    if pmax == 0: pmax = 0.03

    x = array(x)
    yy = pmax * exp( -ka * log(xmax/x)**2 )
    x0 = 1/1.39
    y0 = pmax * exp( -ka * log(xmax/x0)**2 )
    ix = where( x <= x0 )[0]; cx = size(ix)
    if cx > 0: yy[ix] = y0 * ( x[ix]/x0 )**1.7

    return yy


def show_dustem( path='', nh=1e20, fsed='', sw=0, inst='', smdat={}, com='', wref=2.5e2, mbb=[17.75,1.8], npar=0, wext=0.,  
                 xre=[], yre=[], tit='DustEM', cmb=0, cosmo=0, data=0, show=['emis'], freq=0, wn=-1 ):

    if len(path) == 0: 
        print('-----------------------------------------------------------------------------------------------------------------------------------')
        print('def show_dustem( path="", nh=1e20, fsed="", sw=0, inst="", smdat={0}, com='', wext=0, wref=2.5e2, mbb=[17.75,1.8], npar=0,       ')
        print('                 xre=[], yre=[], tit="DustEM", cmb=0, cosmo=0, data=0, show=["emis"], freq=0, wn=-1 )')
        print('-----------------------------------------------------------------------------------------------------------------------------------')
        print()
        print(' returns data dictionary and shows the results of DUSTEM overlaid on diffuse ISM dust observables')
        print()
        print(' PATH   (I): string, path of DustEM directory')
        print(' NH     (I): column density (cm-2) to show emission [Default is 1e20 cm-2]')
        print(' FSED   (I): string, to overlay SED from file FSED (located in path+''/out/'')')
        print(' SW     (I): SW nr. of points for SED smoothing, if 0 no smoothing')
        print(' INST   (I): string array of instrument among AKARI,CAM,DIRBE,HFI,IRAC,IRAS,FIRAS,LFI,MIPS,MIRI,NIRCAM,PACS,SPIRE,SERK,SM79 ')
        print('             WMAP (upper or lowercase). Default is  ["DIRBE","FIRAS","HFI","WMAP"].')
        print(' SMDAT  (O): create or update output dictionary containing model projected on data points ("I_INST" key) ')
        print(' COM    (I): comments for smdat')
        print(' WEXT   (I): normalizes the UV-extinction to 1 at wave WEXT (in microns), then also applied to IR-extinction')
        print(' WREF   (I): get the dust cross-section per H at wave WREF (in microns)')
        print(' MBB    (I): array(2) T and beta for modified BB to get dust opacity from FIRAS')
        print(' NPAR   (I): total number of parameters for chi2 estimate on all data')
        print(' XRE,YRE(I): plot ranges for SED')
        print(' TIT    (I): title for plots')
        print(' CMB    (I): if 1 overlays the CMB')
        print(' COSMO  (I): 1 to plot long wavelengths Planck 2011 data, 2 for Watson+2005 paper.')
        print(' DATA   (I): if DATA=0 (default) only the model SED is shown, DATA=1 only the data, DATA=2 both, DATA=3 model SED and bandfluxes')
        print(' SHOW   (I): string array, defines plot display among "emis", "extuv", "extir", "alb", "sdist", "polext", "polsed", "align".')
        print('             Default is ["emis"]. SHOW=0 no plot')
        print(' FREQ   (I): if 1 frequency (GHz) is shown on top of SED plot')
        print(' WN     (I): array window numbers for plot display.')
        print()
        print('Examples:')
        print('>>> pth = "/Users/lverstra/DUSTEM_LOCAL/dustem4.3" ')
        print('>>> from show_dustem import *')
        print('>>> sm=show_dustem(path=pth)                            # to show SED only ')
        print('!!! show_dustem(path=pth) only will print the (large) sm dictionary')
        print('>>> sm=show_dustem(path=pth,show="extuv")               # SED+UV-extinction ')
        print('>>> sm=show_dustem(path=pth,show=["extir","sdist"])     # SED+IR extinction+size dist. ')
        print('>>> sm=show_dustem(path=pth,show=0)                     # no plot ')
        print('>>> sm=show_dustem(path=pth, inst=["spire"])            # to select or add instrument (see list above)')
        print('>>> sm                                                  # data dictionary with inst data and models')
        print('>>> sm1=show_dustem(path,smdat=sm,inst=[''spire''])   # to add instrument spire fluxes on existing sm')
        print('>>> list(sm)                                            # overall look at dictionary content')
        print('>>> list(sm["M_EMIS"]); list(sm["I_MIPS"]               # to list keys within sm')
        print('>>> sm["M_EMIS"]["UNIT"]                                # to look at particular key values')
        print('>>> sm["I_HFI"]["NAME"]                                 # instrument band names')
        print('>>> sm["I_HFI"]["TRANS"]["Y"][:,1]                      # inst. transmission for 2nd band')
        print('>>> sm["I_HFI"]["YM"][:,1]                              # fluxes of model in 2nd band (last value is sum of all dust types')
        print()
        print()
        print(' Created in IDL @ IAS by L Verstraete and M Compiegne, Spring 2010')
        print(' 2011: LV go to cgs, add band fluxes in SMDAT.')
        print(' 2017: V Guillet and LV add polarization part')
        print(' 2018: ported to python (LV) ')
        print(' 2021: restore band flux SMDAT as a dictionary (LV)')
        print('-----------------------------------------------------------------------------------------------------------------------------------')
        return
    
#
# inits
#
    from scipy import io
    from py4dm import blackbody,chi2
    from band_flux import get_band_flux
    from str_tools import add_inst,add_mod,fil_chi2
    import matplotlib.pyplot as plt
    from astropy.convolution import Box1DKernel, convolve
    import os as os
    import sys as sys

    col = ['black', 'red', 'green', 'blue', 'orange', 'darkorchid', 'magenta','cyan']; ncl = size(col)
    ls = ['-', '--', ':', '-.','--']; nls = size(ls)
    
    if len(inst) == 0:
        inst = ['DIRBE','FIRAS','HFI','SM79','WMAP']
    else:
        if len(inst) != size(inst): inst = [inst]
        inst = [x.upper() for x in inst]
        inst = list(unique(inst)); inst = list(sort(inst))        
    n_inst = size(inst)
    
    if wn < 0: wn = [x for x in range(14)] # absolute display 
    if size(xre) != 2:  # plot ranges for SED (defined according to data ranges)
        if cosmo > 0: xre = [1.,1.e5]
        else: xre = [1.,1.e3] 
    else: xre = array([min(xre),max(xre)])
    if size(yre) != 2:
        if cosmo > 0: yre = [1.e-12,1.e-3]
        else: yre = [1e-8,1e-4] 
    else: yre = array([min(yre),max(yre)])

    if show == 0: show=str(show)
    if len(show) != size(show): show = [show]
    show =  [x.lower() for x in show]
    if show[0] != '0': show = append(show,'emis')  # always show the SED 

# constants 
    na = 6.02e23
    clight = 2.9979246e10  # cm/s

#
# get the SED data
#
# New FarIR-mm SED from F Boulanger
#
    dct = io.readsav( path+'/py/filters_ref_wave.xdr' ) # get ref waves of filters
    wdirbe=dct['wdirbe']; nudirbe=dct['nudirbe']; dnudirbe=dct['dnudirbe']; dwdirbe=dct['dwdirbe']
    wirac=dct['wirac']; nuirac=dct['nuirac']; dwirac=dct['dwirac'] 
    wmips=dct['wmips']; numips=dct['numips']; dwmips=dct['dwmips']
    wpacs=dct['wpacs']; nupacs=dct['nupacs']; dwpacs=dct['dwpacs']; dnupacs=dct['dnupacs']
    wspire=dct['wspire']; nuspire=dct['nuspire']; dwspire=dct['dwspire']
    wlfi=dct['wlfi']; nulfi=dct['nulfi']; dwlfi=dct['dwlfi']; dnulfi=dct['dnulfi']
    whfi=dct['whfi']; nuhfi=dct['nuhfi']; dwhfi=dct['dwhfi']; dnuhfi=dct['dnuhfi']
    wiras=dct['wiras']; nuiras=dct['nuiras']; dwiras=dct['dwiras']
     
    fact_em = 0.77                     # to account for 20% of H to be in ionized form + 3% H2
    to_sed_20 = 1e-17 * (1e20/1.82e18) # MJy/sr->erg/s/cm2/sr and normalization to NH = 10^20 H/cm2
 
    fname = path + '/py/diffuse_ISM_SED.dat'
    f = open(fname,'r')
    lines = f.readlines()
    f.close()
    
    lin = lines[5:162]     # FIRAS
    np = 156
    wfirasf=array([0.0 for x in range(np)]); firasf=array([0.0 for x in range(np)]); ufirasf=array([0.0 for x in range(np)])
    for i in range(np):
        tt=lin[i].split(); wfirasf[i]=tt[0]; firasf[i]=tt[1]; ufirasf[i]=tt[2]
    nufirasf = clight / wfirasf/1.e-4
    firasf   = firasf * nufirasf * fact_em * to_sed_20 
    ufirasf  = ufirasf * nufirasf * to_sed_20 # error
        
    lin = lines[162:166]   # DIRBE 60 to 240
    np = 4
    wdirbef=array([0.0 for x in range(np)]); dirbef=array([0.0 for x in range(np)]); udirbef=array([0.0 for x in range(np)])
    for i in range(np):
        tt=lin[i].split(); wdirbef[i]=tt[0]; dirbef[i]=tt[1]; udirbef[i]=tt[2]
    nudirbef  = clight/ wdirbef/1e-4
    dirbef    = dirbef * nudirbef * fact_em * to_sed_20
    udirbef   = udirbef * nudirbef * to_sed_20 # error
    dwdirbef  = dwdirbe[6:10]/2.

    lin = lines[170:172]   # WMAP
    np = 2
    wwmapf=array([0.0 for x in range(np)]); wmapf=array([0.0 for x in range(np)]); uwmapf=array([0.0 for x in range(np)])
    for i in range(np):
        tt=lin[i].split(); wwmapf[i]=tt[0]; wmapf[i]=tt[1]; uwmapf[i]=tt[2]
    nuwmapf = clight / wwmapf/1.e-4 
    wmapf   = wmapf * nuwmapf * fact_em * to_sed_20 
    uwmapf  = uwmapf * nuwmapf * to_sed_20 # error

# DIRBE Arendt et al (1998) for |b|>25
# numbers from Li & Draine, apj, 2001, 554, 778-802
    wave_arendt  = wdirbe[2:]
    dwave_arendt = dwdirbe[2:] / 2.
    arendt = array( [0.97, 1.11, 7.16, 3.57, 5.30, 18.6, 22.5, 10.1] ) # 10^-26 erg/s/H/sr
    arendt = arendt * 1.e-26 * 1.e20                                   # To erg s-1 cm-2 sr-1 for NH=10^20 H/cm2
    err_arendt = 0.2 * arendt                                          # from Dwek et al. 1997 DIRBE 1Sigma unc = 20%
    nu_arendt = nudirbe[2:] * 1.e9

# Normalization of the Arendt spectrum on the Boulanger 100 microns
    wave_arendt_midir  = wave_arendt[0:4]
    dwave_arendt_midir = dwave_arendt[0:4]
    nu_arendt_midir    = nu_arendt[0:4]
    correl_coeff_midir  = array( [0.00183, 0.00291, 0.0462, 0.0480] )       # for inu
    ucorrel_coeff_midir = array( [0.00001, 0.00003, 0.0001, 0.0002] )       # 
    arendt_midir = correl_coeff_midir * (dirbef[1]*100.*1e-4/clight*1e20)   # in (inu) mjy sr-1
    err_arendt_midir = array( [0.0 for x in range(size(arendt_midir))] ) 
    for i in range(4): 
        tmp=sqrt( ((udirbef[1]*1e2*1e-4/clight*1e20)/(dirbef[1]*1e2*1e-4/clight*1e20))**2.+(ucorrel_coeff_midir[i]/correl_coeff_midir[i])**2. )
        err_arendt_midir[i]  = arendt_midir[i] * tmp
    arendt_midir = arendt_midir / wave_arendt_midir/1e-4*clight/1e20          # to erg s-1 cm-2 sr-1 
    err_arendt_midir = err_arendt_midir / wave_arendt_midir/1e-4*clight/1e20  # to erg s-1 cm-2 sr-1 
    wdirbe_ful = append(wave_arendt_midir,wdirbef)
    dwdirbe_ful = append(dwave_arendt_midir, dwdirbef)
    dirbe_ful  = append(arendt_midir, dirbef)
    udirbe_ful = append(err_arendt_midir, udirbef)

# isocam galactic spectrum
    dct=io.readsav(path+'/py/spectre_gal.xdr')       # wgal in microns, spec_gal in MJy/sr
    wgal=dct('wgal'); spec_gal=dct['spec_gal']; filters=dct['filters'];  
    
    nuisocam =   clight / wgal/1.e-4
    norm = 0.65*0.045/filters[9]                  # normalization for Nh=1e20 cm-2 and I12/I100 =0.045 
    iso = 1e-17 * spec_gal*nuisocam * norm        # galactic mid-IR SED in erg/s/cm2/sr
    tt = blackbody( x=wgal, temp=3e3, unit='w' )
    stars = tt['bb'] * 1e3 * wgal                 # assumed spectrum for stars in cgs
    stars = 1.3e-6 * stars/stars[0]               # normalization of stellar spectrum
    err_cvf = 0.2 
    isocam   = iso - stars
#    isocam = isocam / smp.i_dirbe.ym(4) * arendt_midir[2] # factor is now 1.013027 (1.0235712 before)

# Arome 3.3 microns: Giard et al 1988
    xg = array( [3.3] )
    dxg = 0.05   # assumes a feature width of 0.05 mic
    yg = 1.1e-6 / 4./pi * xg/dxg  # SED erg/s/cm2/sr
    eg = 0.5e-6 / 4./pi * xg/dxg # error

# Modified BB overlaid to dust: default is 17.75 K and beta=1.8  (Miville-Deschênes et al 2013, Planck data)
    np = 1000; wbr = log10( array([20.,1e5]) )
    dwbr = (wbr[1]-wbr[0]) / (np-1)
    lambdaf = 10.**(wbr[0] + arange(np)*dwbr)
    temp1 = mbb[0]; beta = mbb[1]
    lamb_ref = 250.
    qf = (lambdaf/ lamb_ref)**(-beta)
    tt = blackbody( x=lambdaf, temp=temp1, unit='w' )
    bbm = 1e3 * lambdaf * tt['bb']
    sp1 = bbm*qf
    firas_nuinu = firasf
    
# normalize to FIRAS @ wave WREF < wfiras_max and get dust cross-section per H
    wfiras_max = 8e2
    if wref <= wfiras_max: lamb_ref = wref
    tt = list(abs(wfirasf-lamb_ref))
    ifiras = tt.index(min(tt))
    nw = 1
    eps_firas = median(firasf[ifiras-nw:ifiras+nw+1]) / interp(lamb_ref, lambdaf, sp1/qf) / nh
    sp1 = sp1 * median(firasf[ifiras-nw:ifiras+nw+1]) / interp(lamb_ref, lambdaf, sp1)

# add HFI (Planck2011t, Dust in the diffuse interstellar medium and the Galactic halo)
    nu_pep_dism = array([5e3,3e3,857.,545.,353.]) * 1e9 # !!!! includes better IRAS60 and 100
    w_pep_dism = 1e4*clight / nu_pep_dism
    dw_pep_dism = array( [0.0 for x in range(size(nu_pep_dism))] )
    dw_pep_dism[0:2] = array([31.,35.6]) / 2.
    dw_pep_dism[2:]  = w_pep_dism[2:]/3. / 2.
    y_pep_dism = array( [ 1.288, 6.522,5.624, 1.905, 0.465 ] ) * 1e-1  # MJy/sr/1e20cm-2 
    u_pep_dism = array( [ 0.436, 1.473, 1.015, 0.347, 0.100 ] ) * 1e-1
    y_pep_dism = fact_em * y_pep_dism * nu_pep_dism * 1e-17            # MJy/sr/1e20cm-2 to erg/s/cm2/sr
    u_pep_dism = fact_em * u_pep_dism * nu_pep_dism * 1e-17
    y_hfi_dism = y_pep_dism[0:3]; u_hfi_dism = u_pep_dism[0:3]

# add long wavelengths
    if (cosmo == 1) or (cosmo == 2):

        if cosmo == 2:   # Watson et al 2005
            nu=1.e9*array([ 0.408, 1.42, 10.9, 12.7, 14.7, 16.3, 22.8, 33.0, 40.9, 61.3, 93.8, 1250, 2143., 3.e3 ])
            i_jy=array([6.3,7.3,17.,18.7,26.2,32.6,42.3,40.3,33.9,34.7,77.5,9.68e4,1.31e5,6.44e4 ])
            e_jy=array([7.8,2.,0.1,0.4,0.6,1.5,0.2,0.4,0.7,1.8,4.3,7e2,1.25e3,100.])
            beam = 4.9e-4               # steradians
            lref = 140.                 # wave for normalization

        if cosmo == 1:  # Planck 2011p, "New light on AME from spinning dust grains"
                        # nu in GHz and i_jy in Janskys
            nu=array([0.408,0.82,1.42,10.9,12.7,14.7,16.3,22.8,28.5,33.0,40.9,44.1,61.3,70.3,93.8,100.,143.,217.,353.,545.,857.,1250,2143.,3.e3])
#            iplck = where( (nu == 33.) or (nu == 40.9) or (nu == 70.3) or (nu == 100.) or (nu == 143.) or (nu == 217.) )
            nu=1e9*array([0.408,0.82,1.42,10.9,12.7,14.7,16.3,22.8,28.5,33.0,40.9,44.1,61.3,70.3,93.8,100.,143.,217.,353.,545.,857.,1250,2143.,3.e3])
            i_jy=[9.7,9.4,8.0,16.1,20.,28.4,35.8,39.8,38.1,36.7,30.8,28.5,27.4,32.2,63.,78.4,202.,1050.,3060.,1.53e4,4.87e4,9.3e4,1.17e5,5.36e4]
            e_jy=[3.3,4.7,1.7,1.8,2.2,3.1,4.,4.2,4.,3.9,3.3,3.2,3.4,3.9,7.8,15.4,22.,128.,467.,2.09e3,6.11e3,1.29e4,1.45e4,6.67e3] 
            fwhm = 1.81
            beam = pi*(pi/1.8e2 * fwhm/2.)**2 # steradians
            lref = 140.                       # wave for normalization

        x_pep_ame = 1e4*clight/nu
        nc = size(x_pep_ame)
        y_pep_ame = 1.e-23 * nu * i_jy / beam
        e_pep_ame = 1.e-23 * nu * e_jy / beam
        iref = where(wdirbe_ful == lref)[0]; cr=size(iref)
        if cr == 1:
            yic = interp(wdirbe_ful[iref],x_pep_ame,y_pep_ame)
            y_norm = dirbe_ful[iref]/yic
            y_norm = y_norm[0]
        else: y_norm = 1.
        ix = where( x_pep_ame/lref >= 0.9 )[0]; cs=size(ix)
        x_pep_ame=x_pep_ame[ix]; y_pep_ame=y_pep_ame[ix]*y_norm; e_pep_ame=e_pep_ame[ix]*y_norm

#
# plot SED data
#
    if show[0] != '0':
        wnum = wn[0]
        fig = plt.figure(wnum,figsize=(9,6))
        fig.canvas.manager.set_window_title('SHOW_DUSTEM: SED '+str(wnum))
        plt.suptitle(tit)
        xtit = 'Wavelength ($\mu$m)'; xr = xre
        ytit = '$\\nu$ $I_\\nu$ (erg/s/cm$^{2}$/sr)'; yr = yre
        plt.xscale('log'); plt.xlabel(xtit)
        plt.yscale('log'); plt.ylabel(ytit)
        plt.xlim(xr); plt.ylim(yr)

        if (data > 0 and data < 3):  # overlay data
            plt.errorbar(xg,yg,xerr=dxg ,yerr=eg,fmt='+',color='b')
            plt.errorbar(wfirasf,firasf,yerr=ufirasf,fmt=' ',color='k')
            plt.errorbar(wgal,isocam,yerr=isocam*err_cvf,fmt=' ',color='k')
            plt.errorbar(w_pep_dism,y_pep_dism,xerr=dw_pep_dism,yerr=u_pep_dism,fmt='+',color='b') # 60 and 100 from Planck2011t
            plt.errorbar(wdirbe_ful[0:4],dirbe_ful[0:4],xerr=dwdirbe_ful[0:4],yerr=udirbe_ful[0:4],fmt='+',color='b')
            ni = 6 # exclude 60 and 100 from DIRBE (ni=4 to get them back)
            plt.errorbar(wdirbe_ful[ni:],dirbe_ful[ni:],xerr=dwdirbe_ful[ni:],yerr=udirbe_ful[ni:],fmt='+',color='b')
            plt.errorbar(wwmapf,wmapf,yerr=uwmapf,fmt='+',color='b')
            if cosmo != 0: plt.errorbar(x_pep_ame,y_pep_ame,yerr=e_pep_ame,fmt='^',color='b')

        if data == 1:
            plt.plot(lambdaf,sp1,':',color='black') # FIR blackbody
            sp1_firas = interp(wfirasf,lambdaf,sp1)
            firas_chi2 = chi2( firasf, sp1_firas, 3, err=ufirasf )
            print('(W) SHOW_DUSTEM: overlaid modified BB',mbb)
            print('    Chi-square of modified BB to FIRAS: ', firas_chi2)

        if cmb == 1:
            lambda_cmb = 1.e2**( 1.0 + arange(500)*0.01 )
            tt = blackbody(x=lambda_cmb, temp=2.728, unit='w')
            sp_cmb = 1e3 * lambda_cmb *  tt['bb']         # SED in erg/s/cm2/s/sr
            plt.plot(lambda_cmb,sp_cmb,color='black')                   # CMB
            
        if freq == 1:
            ax1 = plt.twiny()  # creates the 2nd x-scale
            #ax1.set_xlabel('Frequency (GHz)')
            plt.xscale('log'); plt.xlabel('Frequency (GHz)')
            xxr = 1e5/array(xre); plt.xlim(xxr)
            plt.xlim(xre)
            
#
# get and plot the model SED
#
    fname = path + '/data/GRAIN.DAT'
    f = open(fname,'r')
    lines = f.readlines()
    f.close()
    print(); print(fname)
    ix=[]
    for i in range(size(lines)):  # to skip comments
        print(lines[i],end='')
        if lines[i][0] != '#': ix.append(i)
    lines = lines[ix[0]:]
    
    r_opt = array(lines[0].split())
    g0 = double(lines[1].split()[0])
    lines = lines[2:]
    ntype = size(lines)
    nsize=arange(ntype); t_opt=['' for x in range(ntype)]; gtype=['' for x in range(ntype)]
    for i in range(ntype):
        nsize[i] = int(lines[i].split()[1])
        t_opt[i] = lines[i].split()[2].lower()
        gtype[i] = lines[i].split()[0]
    nsz_max = max(nsize)
    ipol = [i for i, s in enumerate(t_opt) if 'pol' in s]; cpol = size(ipol) # check for polarization model
    if cpol>0: inst.append('SERK')
    ipol = [i for i, s in enumerate(show) if 'pol' in s]; spol = size(ipol) # check for show polarization

    if npar == 0: npar = ntype  # nr of parameters for chi2
        
    f = open(path+'/out/SED.RES','r')
    lines = f.readlines()
    f.close()
    ix=[]
    for i in range(size(lines)):  # to skip comments
        if lines[i][0] != '#': ix.append(i)
    print(' SED.RES types: '+lines[ix[0]-1], end='')
    lines = lines[ix[0]:]
    ntype_sed = int(lines[0].split()[0])
    if ntype_sed != ntype:
        print('(F) SHOW_DUSTEM: SED.RES & GRAIN.DAT have different NTYPE')
        print('                 data is not from present GRAIN.DAT')
        return
    nlamb = int(lines[0].split()[1]) # nr of wavelengths
    lines = lines[1:]
    xlamb = double([0.0 for x in range(nlamb)])
    sedh = double( [[0.0 for x in range(nlamb)] for y in range(ntype+1)] )
    for i in range(nlamb):
        xlamb[i] = double(lines[i].split()[0])
        sedh[:,i] = double( lines[i].split()[1:] )
    sed = sedh * nh / 4. / pi     # in erg/s/cm2/sr

    if data != 1 and show[0] != '0':
        if sw > 0: k2conv = Box1DKernel(sw)            
        for i in range(ntype+1):    # overlay model SED in erg/s/cm2/sr
            y = sed[i,:]
            if sw > 0: y = convolve(sed[i,:],k2conv)
            linstyl = ls[i%nls]; cl = col[0]
            if i == ntype:
                linstyl='-'; cl=col[1]; gtp = ''
            else: gtp=gtype[i]
            plt.plot(xlamb,y,linstyl,color=cl,label=gtp)
        
        stt = '$G_0$ = '+str('{0:.1e}'.format(g0) )
        plt.annotate(stt,xy=(0.77,0.85),xycoords='figure fraction',color='k')
        stt = '$N_H$ = '+str('{0:.1e}'.format(nh))+' cm$^{-2}$'
        plt.annotate(stt,xy=(0.6,0.85),xycoords='figure fraction',color='k')
        plt.legend(loc=(0.01,0.01))

        if len(fsed) != 0: # read FSED
            fsed = path + '/out/' + fsed
            f = open(fsed,'r')
            lines = f.readlines()
            f.close()
            ix=[]
            for i in range(size(lines)):  # to skip comments
                if lines[i][0] != '#': ix.append(i)
            print(' OSED.RES types: '+lines[ix[0]-1], end='')
            lines = lines[ix[0]:]
            ntype_osed = int(lines[0].split()[0])
            nlambo = int(lines[0].split()[1]) # nr of wavelengths
            lines = lines[1:]
            xlambo = double([0.0 for x in range(nlambo)])
            osedh = double( [[0.0 for x in range(nlambo)] for y in range(ntype_osed+1)] )
            for i in range(nlambo):
                xlambo[i] = double(lines[i].split()[0])
                osedh[:,i] = double( lines[i].split()[1:] )
            osed = osedh * nh / 4. / pi     # in erg/s/cm2/sr

            for i in range(ntype_osed+1):    # overlay model FSED in erg/s/cm2/sr
                y = osed[i,:]
                if sw > 0: y = convolve(osed[i,:],k2conv)
                linstyl = ls[i%nls]; cl = col[0];
                if i == ntype_osed:
                    linstyl='-'; cl='blue'; gtp = ''
                else: gtp=gtype[i]
                plt.plot(xlambo,y,linstyl,color=cl,label=gtp)

#
# fill in SED data and model in SMDAT
# 
# first get model fluxes in band instruments
    unit = 'x(microns) SED(erg/s/cm2/sr)'
    stag = list(smdat)
    if size(stag) == 0:  # create smdat dict with 'M_EMIS' model and band instruments
        mtag='emispol' if cpol>0 else 'emis'
        smdat = get_band_flux( path + '/py/FILTERS', inst=inst, xs=xlamb, ys=sed, mtag=mtag, unit=unit, verb=0 )
    else:             # update smdat with band instrument ('M_EMIS' already present) 
        smdat = get_band_flux( path + '/py/FILTERS', inst=inst, xs=xlamb, ys=sed, unit=unit, smi=smdat, verb=0 )
    if smdat != -1:
        stag = array(list(smdat))
        if com != '': smdat['COM'] = com
    else: stag = []
                
# get diffuse data available here
    # 1st broad band data
    ii = where( stag == 'I_DIRBE')[0]; cic = size(ii)
    if cic>0:
        tt=append([0.,0.],arendt_midir);tt=append(tt,0)
        tt=append(tt,dirbef);tt=append(tt,0.)
        smdat['I_DIRBE']['YD'] = tt
        tt=append([0.,0.],err_arendt_midir);tt=append(tt,0)
        tt=append(tt,udirbef);tt=append(tt,0.)
        smdat['I_DIRBE']['ERR'] = tt
        smdat['I_DIRBE']['NPAR'] = ntype # all grains types
    ii = where( stag == 'I_HFI')[0]; cic = size(ii)
    if cic>0:
        smdat['I_HFI']['YD'] = append(y_hfi_dism,[0.,0.,0.])
        smdat['I_HFI']['ERR'] = append(u_hfi_dism,[0.,0.,0.])
        smdat['I_HFI']['ISEL'] = [1,1,1,0,0,0]
        smdat['I_HFI']['NPAR'] = 2  # carbon and silicates
    ii = where( stag == 'I_WMAP')[0]; cic = size(ii)
    if cic>0:
        smdat['I_WMAP']['YD'] = append([ 0., 0., 0.], wmapf)  # only 61 and 94 GHz bands
        smdat['I_WMAP']['ERR']= append([ 0., 0., 0.], uwmapf)
        smdat['I_WMAP']['ISEL'] = [ 0,0,0,1,1 ]
        smdat['I_WMAP']['NPAR'] = 1  # only spinning grains
    # then spectral data
    ii = where( array(inst) == 'FIRAS')[0]; cic = size(ii)
    if cic>0:
        nband = size(wfirasf)
        smdat = add_inst( nsz=[nband,ntype+1], sf=smdat, tag="FIRAS", photm=0 )
        smdat['I_FIRAS']['UNIT']  = 'x(microns) YM(erg/s/cm2/sr) YD(erg/s/cm2/sr)'
        smdat['I_FIRAS']['X'] = wfirasf 
        smdat['I_FIRAS']['YD'] = firasf
        smdat['I_FIRAS']['ERR'] = ufirasf
        smdat['I_FIRAS']['NPAR'] = 2  # carbon and silicates   
        sd_firas = double([ [0.0 for x in range(nband)] for y in range(ntype+1) ])
        for itp in range(ntype+1): sd_firas[itp,:] = interp( wfirasf, xlamb, sed[itp,:] )
        smdat['I_FIRAS']['YM']= sd_firas

# overlay model band fluxes 
    if ( (data > 1) and (show[0] != '0') ):
        itg = [i for i, s in enumerate(stag) if 'I_' in s]; ctg = size(itg) # finds indices of 'I_' in key list
        for k in range(ctg):  # inst loop
            sstg = list(smdat[stag[itg[k]]])
            itc = [i for i, s in enumerate(sstg) if 'CC' in s]; ctc = size(itc) # finds indices of 'CC' in key list
            if ctc == 1:
                nt = shape(smdat[stag[itg[k]]]['FLX'])[0]
                plt.scatter( smdat[stag[itg[k]]]['X'],smdat[stag[itg[k]]]['YM'][nt-1,:], \
                                 marker='s',s=60,color='none',edgecolor='green',linewidths=2)

#
# get extinction
#
    fname = path + '/out/EXT.RES'
    f = open(fname,'r')
    lines = f.readlines()
    f.close()
    ix=[]
    for i in range(size(lines)):  # to skip comments
        if lines[i][0] != '#': ix.append(i)
    print(' EXT.RES types: '+lines[ix[0]-1],end='')
    lines = lines[ix[0]:]
    ntype_ext = int(lines[0].split()[0])
    if ntype_ext != ntype:
        print('(F) SHOW_DUSTEM: EXT.RES & GRAIN.DAT have different NTYPE')
        print('                 data is not from present GRAIN.DAT')
        return
    nw = int(lines[0].split()[1]) # nr of wavelengths
    if nw != nlamb:
        print('(F) SHOW_DUSTEM: EXT.RES & SED.RES have different nr. of waves')
        print('                 Inconsistent data sets')
        return
    lines = lines[1:]
    stmp = double( [[0.0 for x in range(nlamb)] for y in range(2*ntype+1)] )
    ssca = double( [[0.0 for x in range(nlamb)] for y in range(ntype+1)] )
    sabs = double( [[0.0 for x in range(nlamb)] for y in range(ntype+1)] )
    sext = double( [[0.0 for x in range(nlamb)] for y in range(ntype+1)] )
    alb = double( [[0.0 for x in range(nlamb)] for y in range(ntype+1)] )

    for i in range(nlamb):
#        xlamb[i] = double(lines[i].split()[0])
        stmp[:,i] = double(lines[i].split()[1:])
    xm = 1./xlamb   # 1/micron

    for i in range(ntype):
        sabs[i,:] = stmp[i,:] 
        ssca[i,:] = stmp[ntype+i,:] 
        sext[i,:] = sabs[i,:] + ssca[i,:]
        alb[i,:] = ssca[i,:] / sext[i,:]

    for i in range(nlamb):
        sabs[ntype,i] = sum(sabs[0:ntype,i])
        ssca[ntype,i] = sum(ssca[0:ntype,i])  
        sext[ntype,i] = sum(sext[0:ntype,i])  
        alb[ntype,i] = ssca[ntype,i]/sext[ntype,i]

# fill in extinction model in SMDAT
    ii = where( array(list(smdat)) == 'M_EXT')[0]; cic = size(ii)
    if cic==0:
        mtag='extpol' if cpol>0 else 'ext'
        smdat = add_mod( nsz=[nw,ntype+1], sf=smdat, tag=mtag )
        smdat['M_EXT']['UNIT']  = 'x(microns) Y(1e-21 cm2/H )'
        smdat['M_EXT']['X'] = xlamb
        smdat['M_EXT']['Y'] = sext*1e21
        smdat['M_EXT']['ABS'] = sabs*1e21
        smdat['M_EXT']['SCA'] = ssca*1e21
        smdat['M_EXT']['ALB'] = alb
        
# get RV
    av = interp( 0.55, xlamb, sext[ntype,:] )
    ab = interp( 0.44, xlamb, sext[ntype,:] )
    rv = av / (ab-av)

# get 250 microns emissivity
    eps_a = double([0.0 for x in range(ntype+1)])
    eps_e = double([0.0 for x in range(ntype+1)])
    eps_s = double([0.0 for x in range(ntype+1)])
    for i in range(ntype):
        eps_a[i]=interp(wref,xlamb,sabs[i,:])
        eps_s[i]=interp(wref,xlamb,ssca[i,:])
        eps_e[i]=interp(wref,xlamb,sext[i,:])
    eps_a[ntype] = interp(wref,xlamb,sabs[ntype,:])
    eps_s[ntype] = interp(wref,xlamb,ssca[ntype,:])
    eps_e[ntype] = interp(wref,xlamb,sext[ntype,:])
    print('(W) SHOW_DUSTEM: model dust cross-section @ ',str('{0:.1f}'.format(wref)),'microns per type and total (cm2/H)')
    print(' Abs :',end='')
    for i in range(ntype): print(str('{0:2s}{1:.4e}'.format(' ',1e-21*eps_a[i])),end='')
    print()
    print(' Sca :',end='')
    for i in range(ntype): print(str('{0:2s}{1:.4e}'.format(' ',1e-21*eps_s[i])),end='')
    print()
    print(' Ext :',end='')
    for i in range(ntype): print(str('{0:2s}{1:.4e}'.format(' ',1e-21*eps_e[i])),end='')
    print()
    if wref > wfiras_max:
        print('(W) SHOW_DUSTEM: WREF above longer FIRAS wave, using 250 microns')
        wp = lamb_ref
    else: wp = wref
    print('(W) SHOW_DUSTEM: dust cross-section @ ', str('{0:.2f}'.format(wp)), ' microns from FIRAS')
    print('    using T =',mbb[0], ' and beta =',mbb[1],' sigma(cm2/H) = ',str('{0:.2e}'.format(eps_firas)) )        
    
# get standard Savage & Mathis 79 + Mathis 1990 extinction and fill in data
    fname = path + '/py/mean_ism_ext.dat'
    f = open(fname,'r')
    lines = f.readlines()
    f.close()
    ix=[]
    for i in range(size(lines)):  # to skip comments
        if lines[i][0] != '#': ix.append(i)
    lines = lines[ix[0]:]
    nsm = size(ix)
    x_sm = double([0.0 for x in range(nsm)]); e_sm = double([0.0 for x in range(nsm)])
    for i in range(nsm):
        x_sm[i] = lines[i].split()[0]
        e_sm[i] = lines[i].split()[1]

# fill in SM79 in SMDAT
    ii = where( array(inst) == 'SM79')[0]; cic = size(ii)
    if cic>0:
        nband = nsm
        smdat = add_inst( nsz=[nband,ntype+1], sf=smdat, tag='SM79', photm=0 )
        smdat['I_SM79']['UNIT']  = 'x(microns) YM(1e-21 cm2/H ) YD(1e-21cm2/H)'
        smdat['I_SM79']['X'] = x_sm
        smdat['I_SM79']['YD'] = e_sm*1e21
        smdat['I_SM79']['ERR'] = 0.2*e_sm*1e21
        smdat['I_SM79']['NPAR'] = ntype  # all grain types
        edm = double([ [0.0 for x in range(nband)] for y in range(ntype+1) ])
        for itp in range(ntype+1): edm[itp,:] = interp( x_sm, xm, sext[itp,:] )
        smdat['I_SM79']['YM']= edm
      
#
# plot extinction
#
# first visible-UV
#
    cuv = 0; cir = 0
    ip = where( show == 'extuv')[0]; cuv = size(ip)
    ip = where( show == 'extir')[0]; cir = size(ip)
    
    yscl = 1.0  # get normalization
    if wext > 0:
        ei = interp(wext,xlamb,sext[ntype,:])
        if ei != 0.: yscl = 1.0 / ei
    
    if show[0] != '0' and cuv > 0:
        wnum = wn[1]
        fig = plt.figure(wnum,figsize=(6,5))
        fig.canvas.set_window_title('SHOW_DUSTEM: UV Extinction '+str(wnum))
        plt.suptitle(tit)
        xtit = 'x ($\mu$m$^{-1}$)'; xr = [0,11] 
        yr=[ 0, 2.5 ]
        ytit='$\sigma_{ext}$ (10$^{-21}$ cm$^2$ per H)'
        
        if wext > 0:
            yr = yscl*array([min(sext[ntype,:]),max(sext[ntype,:])])
            ytit='Normalized $\sigma_{ext}$ '
            print('(W) SHOW_DUSTEM: extinction normalized to 1 at ',wext, ' microns yscl=',str('{0:.2e}'.format(yscl)))

        plt.xscale('linear'); plt.yscale('linear')
        plt.xlabel(xtit); plt.ylabel(ytit)
        plt.xlim(xr); plt.ylim(yr)
        plt.plot(xm,sext[ntype,:]*yscl,color='red',label='')
        if data != 0: plt.plot(1./x_sm, e_sm*1e21*yscl,'+')
        for i in range(ntype):
            cl='k'; gtp=gtype[i]; lins=ls[i%nls]
            plt.plot(xm,sext[i,:]*yscl,lins,color=cl,label=gtp)
        stt = '$R_V$ = '+str('{0:.1f}'.format(rv) )
        plt.annotate(stt,xy=(0.77,0.84),xycoords='figure fraction',color='k')
        plt.legend(loc=(0.01,0.7))
        
# plot IR extinction
    if show[0] != '0' and cir > 0:
        wnum = wn[2]
        fig = plt.figure(wnum,figsize=(6,5))
        fig.canvas.set_window_title('SHOW_DUSTEM: Extinction IR '+str(wnum))
        plt.suptitle(tit)
        xtit = '$\lambda$ ($\mu$m)'; xr = [1,400] 
        yr = [1e-6,1.]
        if wext > 0:
            yr = yscl*array([min(sext[ntype,:]),max(sext[ntype,:])])
            ytit='Normalized $\sigma_{ext}$'
            if cuv == 0:
                print('(W) SHOW_DUSTEM: extinction normalized to 1 at ',wext, ' microns yscl=',str('{0:.2e}'.format(yscl)))
        else: ytit='$\sigma_{ext}$ (10$^{-21}$ cm$^2$ per H)'
        plt.xscale('log'); plt.yscale('log')
        plt.xlabel(xtit); plt.ylabel(ytit)
        plt.xlim(xr); plt.ylim(yr)
        plt.plot(xlamb,sext[ntype,:]*yscl,color='red',label='')
        if data != 0: plt.plot(x_sm, e_sm*1e21*yscl,'+')
        for i in range(ntype):
            cl='k'; gtp=gtype[i]; lins=ls[i%nls]
            plt.plot(xlamb,sext[i,:]*yscl,lins,color=cl,label=gtp)
        stt = '$R_V$ = '+str('{0:.1f}'.format(rv) )
        plt.annotate(stt,xy=(0.77,0.84),xycoords='figure fraction',color='k')
        plt.legend(loc=(0.01,0.01))

# and the albedo
    ip = where( show == 'alb')[0]; cp = size(ip)
    if show[0] != '0' and cp > 0:
        wnum = wn[3]
        fig = plt.figure(wnum,figsize=(6,5))
        fig.canvas.set_window_title('SHOW_DUSTEM: Albedo '+str(wnum))
        plt.suptitle(tit)
        xtit = '1/$\lambda$ ($\mu$m$^{-1}$)'; xr = [0,11] 
        ytit='Albedo'; yr=[ 0, 1.1]
        plt.xscale('linear'); plt.yscale('linear')
        plt.xlabel(xtit); plt.ylabel(ytit)
        plt.xlim(xr); plt.ylim(yr)
        plt.plot(xm,alb[ntype,:],color='red',label='')
        for i in range(ntype):
            cl='k'; gtp=gtype[i]; lins=ls[i%nls]
            plt.plot(xm,alb[i,:]*yscl,lins,color=cl,label=gtp)
        stt = '$R_V$ = '+str('{0:.1f}'.format(rv) )
        plt.annotate(stt,xy=(0.77,0.84),xycoords='figure fraction',color='k')
        plt.legend(loc=(0.78,0.7))
        
#
#  get the size distribution and plot
#
    ip = where( r_opt == 'sdist')[0]; csd = size(ip)
    if (csd > 0):
        fname = path + '/out/SDIST.RES'

        if os.path.isfile(fname): # test if SDIST.RES present
            f = open(fname,'r'); lines = f.readlines(); f.close()
            ix=[]
            for i in range(size(lines)):  # to skip comments
                if lines[i][0] != '#': ix.append(i)
            lines = lines[ix[0]:]
            ntype_sdist = int(lines[0].split()[0])             
            lines = lines[1:]; iz = 0; cgood = 0         
            for i in range(ntype): 
                ntz = int(lines[iz].split()[1])
                iz = iz + ntz + 1
                if ntz == nsize[i]: cgood = cgood + 1   

            if (ntype_sdist == ntype) and (cgood == ntype): # test if SDIST.RES is from GRAIN.DAT model             
                ax = double([[0.0 for x in range(nsz_max)] for x in range(ntype)])
                ava = double([[0.0 for x in range(nsz_max)] for x in range(ntype)])
                iz = 1
                for i in range(ntype):
                    ttx=[]; tty=[]
                    for j in range(iz,iz+nsize[i]):
                        ttx.append(lines[j].split()[0])
                        tty.append(lines[j].split()[1])
                    ax[i,0:nsize[i]] = double(ttx)
                    ava[i,0:nsize[i]] = double(tty)
                    iz = iz + nsize[i] + 1

# fill in size distribution in SMDAT
                ii = where( array(list(smdat)) == 'M_SDIST')[0]; cic = size(ii)
                if cic==0:
                    mtag = 'sdistpol' if cpol>0 else 'sdist'
                    smdat = add_mod( nsz=[nsz_max,ntype], sf=smdat, tag=mtag )
                    smdat['M_SDIST']['UNIT']  = 'a(nm) dV/dlog a(1e-29 cm3/H )'
                    smdat['M_SDIST']['A'] = ax*1e7
                    smdat['M_SDIST']['AVA'] = ava*1e29
                    
                ip = where( show == 'sdist')[0]; cd = size(ip) # plot size distribution
                if (show[0] != '0') and (cd > 0):
                    wnum = wn[4]
                    fig = plt.figure(wnum,figsize=(6,5))
                    fig.canvas.set_window_title('SHOW_DUSTEM: Size distribution '+str(wnum))
                    plt.suptitle(tit)
                    yscl = 1e29
                    xtit = 'a (nm)'; xr = [0.1,1e4]
                    ytit='$n_H^{-1}$ $a^4$ d$n$/d$a$ (10$^{-29}$ cm$^{3}$/H) '; yr=[1e-2,1e3] 
                    plt.xscale('log'); plt.yscale('log')
                    plt.xlabel(xtit); plt.ylabel(ytit)
                    plt.xlim(xr); plt.ylim(yr)
                    for i in range(ntype):
                        cl='k'; gtp=gtype[i]; lins=ls[i%nls]
                        ip = where(ax[i,:]>0)[0]
                        plt.plot(ax[i,ip]*1e7,ava[i,ip]*yscl,lins,color=cl,label=gtp)
                    plt.legend(loc=(0.01,0.7))

            elif (ntype_sdist != ntype) and (cgood != ntype): print('(F) SHOW_DUSTEM: SDIST.RES not from GRAIN.DAT')
            
        elif os.path.isfile(fname): print('(W) SHOW_DUSTEM: no SDIST.RES found ')
                
    elif csd == 0: print('(W) SHOW_DUSTEM: GRAIN.DAT model does not include SDIST run keyword')
                    
#
# get polarized data and plot
#
    if cpol == 0 and spol > 0: print('(W) SHOW_DUSTEM: GRAIN.DAT model does not include POL keyword')
        
    if cpol>0:  # with a polarized model
        
        fname = path + '/out/SED_POL.RES'       
        if os.path.isfile(fname): # test if SED_POL.RES present
            f = open(fname,'r'); lines = f.readlines(); f.close()
            ix=[]
            for i in range(size(lines)):  # to skip comments
                if lines[i][0] != '#': ix.append(i)
            print(' SED_POL.RES types: '+lines[ix[0]-1],end='')
            lines = lines[ix[0]:]
            ntype_polsed = int(lines[0].split()[0])
            nw = int(lines[0].split()[1]) # nr of wavelengths
         
            if ntype_polsed == ntype and nw == nlamb: 
                lines = lines[1:]
                sedh_p = double( [[ 0.0 for x in range(nlamb)] for y in range(ntype+1)]) 
                for i in range(nlamb):
                    #xlamb[i] = double(lines[i].split()[0])
                    sedh_p[:,i] = double(lines[i].split()[1:])
                sed_p = sedh_p * nh / 4. / pi # in erg/s/cm2/sr

# fill in polarized SED in SMDAT
                smdat['M_EMIS']['YP'] = sed_p

                ip = where( show == 'polsed')[0]; c_polsed = size(ip)
                if (show[0] != '0') and (c_polsed > 0):
                    wnum = wn[5]
                    fig = plt.figure(wnum,figsize=(6,5))
                    fig.canvas.set_window_title('SHOW_DUSTEM: Polarized SED '+str(wnum))
                    plt.suptitle(tit)
                    xtit = 'Wavelength ($\mu$m)'; xr = [10.,1e4]
                    ytit= '$\\nu$ $P_\\nu$ (erg/s/cm$^2$/sr)'; yr = array([1e-5*max(sed_p[ntype,:]),5*max(sed_p[ntype,:])])
                    plt.xscale('log'); plt.yscale('log')
                    plt.xlabel(xtit); plt.ylabel(ytit)
                    plt.xlim(xr); plt.ylim(array(yr))
                    if freq == 1:
                        ax1 = plt.twiny()  # creates the 2nd x-scale
                        #ax1.set_xlabel('Frequency (GHz)')
                        plt.xscale('log'); plt.xlabel('Frequency (GHz)')
                        xxr = 1e5/array(xr); plt.xlim(xxr)
                        plt.xlim(xr)
        
                    plt.plot(xlamb,sed_p[ntype,:],color='red',label='')
                    for i in range(ntype):
                        cl='k'; gtp=gtype[i]; lins=ls[i%nls]
                        plt.plot(xlamb,sed_p[i,:]*yscl,lins,color=cl,label=gtp)
                    plt.legend(loc=(0.01,0.01))

#
#  Polarization fraction in emission
#
                    wnum = wn[6]
                    fig = plt.figure(wnum,figsize=(6,5))
                    fig.canvas.set_window_title('SHOW_DUSTEM: P/I '+str(wnum))
                    plt.suptitle(tit)
                    xtit = 'Wavelength ($\mu$m)'; xr = [10.,1e4]
                    ytit= '$P/I$'; yr=[ 0., 0.4]
                    plt.xscale('log'); plt.yscale('linear')
                    plt.xlabel(xtit); plt.ylabel(ytit)
                    plt.xlim(xr); plt.ylim(array(yr))
                    if freq == 1:
                        ax1 = plt.twiny()  # creates the 2nd x-scale
                        #ax1.set_xlabel('Frequency (GHz)')
                        plt.xscale('log'); plt.xlabel('Frequency (GHz)')
                        xxr = 1e5/array(xr); plt.xlim(xxr)
                        plt.xlim(xr)
            
                    plt.plot(xlamb,sed_p[ntype,:]/sed[ntype,:],color='red')
                    for i in range(ntype):
                        cl='k'; gtp=gtype[i]; lins=ls[i%nls]
                        i0 = where(sed[i,:] > 0)[0]
                        plt.plot(xlamb[i0],sed_p[i,i0]/sed[i,i0],lins,color=cl,label=gtp)
                    plt.legend(loc=(0.01,0.8))

                    plt.figure(wn[0])  # overplot polarized SED on SED 
                    plt.plot(xlamb,sed_p[ntype,:],color='green')

            else: print('(W) SHOW_DUSTEM: SED_POL.RES & SED.RES from different models')
            
        else: print('(W) SHOW_DUSTEM: no SED_POL.RES found ')            

#
# get the polarised extinction
#
        fname = path + '/out/EXT_POL.RES'
        if os.path.isfile(fname): # test if EXT_POL.RES present
            f = open(fname,'r'); lines = f.readlines(); f.close()
            ix=[]
            for i in range(size(lines)):  # to skip comments
                if lines[i][0] != '#': ix.append(i)
            print(' EXT_POL.RES types: '+lines[ix[0]-1],end='')
            lines = lines[ix[0]:]
            ntype_extpol = int(lines[0].split()[0])
            nw = int(lines[0].split()[1])
            if ntype_extpol == ntype and nw == nlamb:
                lines = lines[1:]
                spabs = double( [[ 0.0 for x in range(nlamb)] for y in range(ntype+1)])
                spsca = double( [[ 0.0 for x in range(nlamb)] for y in range(ntype+1)]) 
                for i in range(nlamb):
                    #xlamb[i] = double(lines[i].split()[0])
                    spabs[0:ntype,i] = double(lines[i].split()[1:ntype+1])
                    spsca[0:ntype,i] = double(lines[i].split()[ntype+1:2*ntype+1])
                    spabs[ntype,i] = sum(spabs[:,i])
                    spsca[ntype,i] = sum(spsca[:,i])

# get the Serkowski model
                ix = where(x_sm <= 7.0)[0]; csm = size(ix)  # keep wave below 7 microns
                if csm > 0:
                    xx = 1./x_sm[ix]
                    xe = e_sm[ix]
                ii = argsort(xx)
                xx = xx[ii]; xe = xe[ii]
                fpol = serkowski(x=xx)

# fill in SMDAT        
                ii = where( array(inst) == 'SERK')[0]; cic = size(ii)
                if cic>0:
                    nband = nsm
                    smdat = add_inst( nsz=[nband,ntype+1], sf=smdat, tag='SERK', photm=0 )
                    smdat['I_SERK']['UNIT']  = 'x(1/microns) YM(erg/s/cm2/sr) YD(erg/s/cm2/sr)'
                    smdat['I_SERK']['X'] = xx
                    smdat['I_SERK']['YD'] = fpol
                    smdat['I_SERK']['ERR'] = 0.2*fpol
                    smdat['I_SERK']['NPAR'] = 2  # carbon and silicates
                    sdm = double([ [0.0 for x in range(nband)] for y in range(ntype+1) ])
                    for itp in range(ntype+1): sdm[itp,:] = interp( x_sm, xm, (spabs[ntype,:]+spsca[ntype,:]) )
                    smdat['I_SERK']['YM']= sdm
                smdat['M_EXT']['ABS_P']= spabs
                smdat['M_EXT']['SCA_P']= spsca
                
# Serkowski type plot
                ip = where( show == 'polext')[0]; c_polext = size(ip)
                if (show[0] != '0') and (c_polext > 0):
                    wnum = wn[7]
                    fig = plt.figure(wnum,figsize=(6,5))
                    fig.canvas.set_window_title('SHOW_DUSTEM: Serkowski plot '+str(wnum))
                    plt.suptitle(tit)
                    xtit = '1/$\lambda$ ($\mu$m$^{-1}$)'; xr = [0.2,10] 
                    ytit= '$\sigma_{pol}$ (10$^{-23}$ cm$^2$/H)'; yr=[ 0.1, 5]
                    plt.xscale('log'); plt.yscale('log')
                    plt.xlabel(xtit); plt.ylabel(ytit)
                    plt.xlim(xr); plt.ylim(array(yr))
                    yscl = 1e23; yscl2 = 1e2
        
                    plt.plot(xx,yscl*fpol/5.8e21*3.1,'+')
                    plt.plot(1./xlamb,yscl2*(spabs[ntype,:]+spsca[ntype,:]),color='red',label='')
                    for i in range(ntype):
                        cl='k'; gtp=gtype[i]; lins=ls[i%nls]
                        plt.plot(1/xlamb,yscl2*(spabs[i,:]+spsca[i,:]),lins,color=cl,label=gtp)
                    plt.legend(loc=(0.01,0.8))
        
#
#  Polarization fraction in extinction
#
                    wnum = wn[8]
                    fig = plt.figure(wnum,figsize=(6,5))
                    fig.canvas.set_window_title('SHOW_DUSTEM: P/tau '+str(wnum))
                    plt.suptitle(tit)
                    xtit = 'Wavelength ($\mu$m)'; xr = [0.1,1e4]
                    ytit= '$P/\\tau$'; yr=[ 0., 0.5]
                    plt.xscale('log'); plt.yscale('linear')
                    plt.xlabel(xtit); plt.ylabel(ytit)
                    plt.xlim(xr); plt.ylim(array(yr))
                    yscl = 1.
                    if freq == 1:
                        ax1 = plt.twiny()  # creates the 2nd x-scale
                        #ax1.set_xlabel('Frequency (GHz)')
                        plt.xscale('log'); plt.xlabel('Frequency (GHz)')
                        xxr = 1e5/array(xr); plt.xlim(xxr)
                        plt.xlim(xr)
        
                    plt.plot(xlamb,yscl*(spabs[ntype,:]+spsca[ntype,:])/sext[ntype,:],color='red')
                    for i in range(ntype):
                        cl='k'; gtp=gtype[i]; lins=ls[i%nls]
                        i0 = where(sext[i,:] > 0)[0]
                        plt.plot(xlamb[i0],yscl*(spabs[i,i0]+spsca[i,i0])/sext[i,i0],lins,color=cl,label=gtp)
                    plt.legend(loc=(0.01,0.8))

            else: print('(W) SHOW_DUSTEM: EXT_POL.RES & SED.RES from different models')

        else: print('(W) SHOW_DUSTEM: no EXT_POL.RES found ')

#
# Alignment function
#
        fname = path + '/data/ALIGN.DAT'
        if os.path.isfile(fname): # test if ALIGN.DAT present
            f = open(fname,'r'); lines = f.readlines(); f.close()
            ix=[]
            for i in range(size(lines)):  # to skip comments
                if lines[i][0] != '#': ix.append(i)
            lines = lines[ix[0]:]
            flags = lines[0]; anisG0 = lines[1]
            align_type = lines[2].split()[0]
            if align_type == 'par':  # get parametric alignment law
                athresh = double(lines[2].split()[1])
                pstiff = double(lines[2].split()[2])
                plev = double(lines[2].split()[3])
                a1 = 1e-3; a2 = 10.; na = 100.
                aa = a1 * exp( arange(na)*log(a2/a1)/(na-1.) )  # grain radius in microns
                fpol = 0.5 * plev * (1. + tanh(log(aa/athresh)/pstiff) )

            if csd > 0:
                smdat['M_SDIST']['AI'] = aa; smdat['M_SDIST']['FI'] = fpol; 
                
            ip = where( show == 'align')[0]; c_align = size(ip)
            if (show[0] != '0') and (c_align > 0):
                wnum = wn[9]
                fig = plt.figure(wnum,figsize=(6,5))
                fig.canvas.set_window_title('SHOW_DUSTEM: Alignement function '+str(wnum))
                plt.suptitle(tit)
                xtit = 'Grain radius ($\mu$m)'; xr = [a1,a2]
                ytit= 'Alignment efficiency'; yr=[ 0., 1.1]
                plt.xscale('log'); plt.yscale('linear')
                plt.xlabel(xtit); plt.ylabel(ytit)
                plt.xlim(xr); plt.ylim(array(yr))
                yscl = 1.
        
                plt.plot(aa,fpol,color='red')

        else: print('(W) SHOW_DUSTEM: no ALIGN.DAT found ')

#
# get all the chi2
#
    smdat = fil_chi2( smdat ) 
    stag = list(smdat)
    itx = [i for i, s in enumerate(stag) if 'I_' in s]; ctx = size(itx) 
    print('Chi-square of model to instrument - nr of parameters')
    for ii in range(ctx): print(str('{0:10s}'.format(stag[itx[ii]]))+'  : ',str('{0:.3e}'.format(smdat[stag[itx[ii]]]['CHI2'])),' - ', \
                                str('{0:3d}'.format(smdat[stag[itx[ii]]]['NPAR'])) )
    print('Chi-square of model to all data - nr of parameters: ', str('{0:.3e}'.format(smdat['CHI2'])),' - ', str('{0:3d}'.format(smdat['NPAR'])) )
            
            
#
# close plotting 
#
    if show[0] != '0':
        plt.show()
        plt.close()

    return smdat
