
from numpy import *
import matplotlib.pyplot as plt

def blackbody( x=0, temp=0., unit='F', wdil=1., norm=0, chk=0 ):
    
    if size(x) == 1:
        print('------------------------------------------------------------------------------------------')
        print('def blackbody( x=x, temp=temp, unit="F", wdil=1, norm=0, chk=0 )')
        print('------------------------------------------------------------------------------------------')
        print(' computes a (or a sum of) blackbody brightness(es) B(x,T) in W/m2/(x unit)/sr ')
        print(' X	  (I): blackbody abscissa in cm-1, GHz, microns or eV')
        print(' TEMP  (I): blackbody temperature (can be an array)')
        print(' WDIL  (I): dilution factor for each TEMP')
        print(' UNIT  (I): blackbody x unit "K" is B_nu (W/m2/cm-1/sr),    x in cm-1 ')
        print('                             "F" is B_nu (W/m2/Hz/sr),      x in GHz')
        print('	                            "W'' is B_lambda (W/m2/um/sr), x in microns')
        print('	 	  	                    "E" is B_E (W/m2/eV/sr), x in eV')
        print(' NORM  (I): overall normalizing factor, returns NORM*BB/MAX(BB)')
        print(' CHK   (I): 1 to get comments')
        print(' CHECK (O): check energy conservation for the BB (power is sigma*T^4)')
        print(' G0    (O): scaling factor for integrated 5-13.6 eV flux in ISRF unit.')
        print(' CHI   (O): scaling factor @ 1000 A in ISRF unit (1.6e-3 erg/cm2/s)')
        print(' Returns a dictionary')
        print('')
        print(' Created Aug. 2009, L. Verstraete, IAS')
        print(' Force the WCUT point, May 2011, LV')
        print(' ported to python, May 2018, LV')
        print('------------------------------------------------------------------------------------------')
        return

    nx = size(x); nt = size(temp); nw = size(wdil)
    if nw == 0: wdil = [1.0 for x in range(nt)]
    else:
        if nw != nt:
            print('(F) BLACKBODY: wdil and temp have different sizes')
            return
    if nt == 1:
        d = str(type(temp))
        if d.find('list') == -1: temp = [temp]
    if nw == 1:
        d = str(type(wdil))
        if d.find('list') == -1: wdil = [wdil]
    temp = array(temp); wdil = array(wdil)
    if len(unit) != 0: unit = unit.upper()
    if unit != 'K' and unit != 'F' and unit != 'W' and unit != 'E':
        print('(W) BLACKBODY: not a valid BB unit --> make a B_nu')
        unit = 'F'

    xpi = 3.1415926535897932385
    clight = 2.9979246e+08          # m/s
    kb = 1.3806503e-23
    hp = 6.62606876e-34
    hck = hp*clight / kb
    ze  = 1.60217653e-19           # electron charge
    hev = hp/ze                    # h in eV unit

    xly = 12.4
    isrf0 = 1.6e-6         # 4*pi*nu*I_nu (W/m2) in solar neighborhood

#
# loop on temperatures
# 
    x = double(x)
    bbt = 0.
    acheck = 0.

    for j in arange(nt):

        xi = array( [6., 13.6] )
        if unit == 'K':
            be = exp( -(1e2*x) * hck / temp[j] )
            bb = be * (1e2*x)**3 / (1.0 - be)
            bb = bb * 2. * hp * clight**2 
            bb = bb * 1e2         # m-1 to cm-1
            xi = xi / hev/clight/1e2   
            x1 = xly / hev/clight/1e2 
        elif unit == 'F':
            be = exp( -(1e9*x) * hp/kb/ temp[j] )
            bb = be * (1e9*x)**3 / (1.0 - be)
            bb = bb * 2. * hp / clight**2
            xi = xi / hev / 1e9
            x1 = xly / hev / 1e9
        elif unit == 'W':
            be = exp( -hp*clight*1e6/kb/x / temp[j] )
            bb = be / (1e-6*x)**5 / (1.0 - be)
            bb = bb * 2. * hp * clight**2 
            bb = 1e-6 * bb        # m-1 to micron-1
            xi = hev*clight*1e6 / xi
            x1 = hev*clight*1e6 / xly
        elif unit == 'E':
            be = exp( -(ze*x)/kb / temp[j] )
            bb = be * (ze*x)**3 / (1.0 - be)
            bb = bb * 2. / hp**3 / clight**2 
            bb = bb * ze          # J-1 to eV-1
            xi = xi
            x1 = xly

        ck = 0.
        for jx in range(nx-1):
            ck = ck + (bb[jx+1]+bb[jx])*(x[jx+1]-x[jx])

        ck = (xpi/2)*ck / 5.6697e-8 / temp[j]**4  

        if chk == 1:
            print( '(W) BLACKBODY: integrated BB('+str(temp[j])+
                   ' K) is '+str('{0:2s}{1:.4e}'.format('',ck))+ ' * Sigma*T^4' )
        acheck = append(acheck, ck)
    
        bbt = bbt + bb*wdil[j] # endfor
 
    check = acheck[1:]


# get G0
    ig = where( (x >= min(xi))*(x <= max(xi)) )[0]; cg = len(ig)
    if cg > 0:
        xx = x[ig]
        yy = xpi*x[ig]*bbt[ig]
        s1 = sum( (yy[1:cg-1]+yy[0:cg-2])*(xx[1:cg-1]-xx[0:cg-2])/2.0 )
        g0 = s1/isrf0
    else: g0 = 0.

# get chi
    ig = where( abs(2.*(x-x1)/(x+x1)) <= 0.1)[0]
    if len(ig) > 0:
        xx1 = abs(x[ig]-x1)
        iig = where(xx1 == min(xx1))[0]
        chi = xpi * x[ig[iig]] * bbt[ig[iig]] / isrf0
    else: chi = 0

    if norm > 0: bbt = norm * bbt/max(bbt)
                                                
    return {'x':x,'bb':bbt,'unit':unit,'check':check,'g0':g0,'chi':chi }      # end blackbody

    
def chi2( y, model, npar, err=0 ):
# returns the chi-square value of fit MODEL to data Y
#
# NPAR (I): nr of parameters in the fit
# ERR  (I): error of each data point. Default is ERR=0.1*Y

  ny = size(y)
  if size(err) == 0 or sum(err) == 0:
     err = 0.1*y
     print('(W) CHI2: error missing, set to 10 %')

  ndof = ny - npar              # nr of degrees of freedom
  chi = sum( ((y-model)/err)**2 )
  if ndof > 0: chi = chi / ndof

  return chi
