from numpy import *
import matplotlib.pyplot as plt

#--------------------------------------------------------------------------------------------------------------------
def dm4cldy( files='',dm=['DustEM/','tt'],yscl=1.,smod=1,wex=[0.1,0.2], tit='',xl=[], yl=[],show=1,chk=0 ):
#--------------------------------------------------------------------------------------------------------------------

 if len(files) == 0: 
    print('---------------------------------------------------------------------------------------------------------------------')
    print('def dm4cldy( files="",dm=dm,yscl=1.,smod=1,wex=[0.1,0.2], tit="",xl=[],yl=[],show=1 )')
    print('---------------------------------------------------------------------------------------------------------------------')
    print()
    print(' Generates a grain opacity file for Cloudy from DustEM data')
    print(' - choose a relevant RFI file in Cloudy /data: will be used for *high-energy extrapolation* and material type ')
    print('   (1:carbonaceous, 2:silicate, 3:PAH (1990s), 6:PAH DL01 & 07, see vanHoof+2004)')
    print(' - define dust size dist file <tt.szd> for Cloudy (see ex. in Cloudy/data). !!! Check that all sizes are within  ')
    print('   bounds of DustEM sizes ')
    print(' - with Cloudy <compile grains NSB bins "be1-amcarb.rfi" "tt.szd"> generates "be1-amcarb_tt.opc". ')
    print('   This provides the format of opacity file: change to "tt.opc" ')
    print(' - Write cloudy opacity "tt.opc" file with present <dm_4_cldy.py>')
    print(' - set correct values to grain molecule and grain atom weights in "tt.opc" and set corresponding element abundances ')
    print()
    print(' Returns dictionary  {"acy"[j,6],"xw","a_cs"[i,j],"s_cs"[i,j],"g"[i,j]} ')
    print('                        i:wave [0,nw], j:size bin [0,nsb+1] (j=nsb+1 is sum of all bins) ')
    print()
    print(' ACY format for each size bin i ')
    print('     acy[i,0] # average grain radius <a^3>/<a^2>,        (cm)     ')
    print('     acy[i,1] # average grain area <4pi*a^2>,            (cm^2)   ')
    print('     acy[i,2] # average grain volume <4/3pi*a^3>,        (cm^3)   ')
    print('     acy[i,3] # total grain radius Int(a) per H,         (cm/H)   ')
    print('     acy[i,4] # total grain area Int(4pi*a^2) per H,     (cm^2/H) ')
    print('     acy[i,5] # total grain volume Int(4/3pi*a^3) per H, (cm^3/H) ')
    print()
    print(' FILES   (I): array(2) with the input Cloudy opacity file FILES(0) used for definition of size dist. and waves while ')
    print('              the output file for Cloudy is FILES(1).')
    print(' DM      (I): string array(2), dm[0] is path to DustEM data, dm[1] is DustEM grain type ')
    print(' YSCL    (I): scaling factor for the full cross-section (to test various sig_IR).')
    print(' SMOD    (I): plot cross-sections every SMOD sizes ')
    print(' WEX     (I): array giving wave range in cm to perform power law extrapolation at low energy')
    print(' XL      (I): X-axis limits in microns for plot of cross-sections')
    print(' YL      (I): Y-axis limits in Mb/H (1e-18 cm2/H) ')
    print(' SHOW    (I): if 1, plot data in cloudy file')
    print(' CHK     (I): if 1 plots Q & g wrt size, if 2 adds original data in dots ')
    print()
    print('Examples:')
    print('>>> from dm4cldy import *')
    print('>>> pth = "/Users/lverstra/cloudy/c17.02/data/"; fls = [pth+"be1-amcarb_cp_themis_18.opc",pth+"cm20_cp_themis_18.opc"] ')
    print('>>> dm = ["/Users/lverstra/DUSTEM_LOCAL/dustem4.3_wk/oprop/","CM20"]')
    print('>>> tt = dm4cldy( files=fls,dm=dm )  # for C grains')
    print()
    print('Written by L. Verstraete (IAS, Spring 2022)')
    print('---------------------------------------------------------------------------------------------------------------------')
    return

#
# inits & checks
#
 mwg = 1.42         # mean gas molecular weight in amu
 navo = 6.02e23     # Avogadro nr
 hbc = 197.3269805  # MeV.fermi
 ryd = 13.6057      # Rydberg in eV
 sep = ' '          # separator to extract substrings

 if size(files) != 2:
     print(' (F): files ill-defined')
     return 0
 else:
     for i in range(2): files[i] = str(files[i])
         

#
# read cloudy file
#
 print('Reading '+files[0])
 f = open(files[0],'r')
 lines = f.readlines()
 f.close()
 
 i = 0
 while '# number of size distr. bins' not in lines[i]: i = i+1
 nsb = int(lines[i].split()[0])
 nw = int(lines[i-1].split()[0])

# open output file, write header and get next headers
 fo = open(files[1],'w')
 io = 0
 while '# anu' not in lines[io]: io = io+1
 for ii in range(io+2): fo.write(lines[ii])
 nh = 3; nhl = 4; hdr = [['' for x in range(nhl)] for y in range(nh)]
 for ii in range(nh):
     io = io+1
     while '# anu' not in lines[io]: io = io+1
     hdr[ii] = lines[io-2:io+2]
     
# get size dist.
 i = i+1
 acy = double( [[0.0 for x in range(6)] for y in range(nsb)] )
 tmp = array(lines)
 lb = 8; ix = arange(lb+1)   # block of 8 lines to be read NSB times
 for s in arange(nsb):
     for nb in 2+arange(6):
         ic = i+lb*s+nb
         acy[s,nb-2] = double(tmp[ic].split()[0])
 print(' nsb = '+str(nsb)+' sizes found ['+str(acy[0,0])+','+str(acy[nsb-1,0])+'] cm')

 i = 0
 while '# size distribution function' not in lines[i]: i = i+1
 nsbc = int(lines[i+2].split()[0])
 acyc = double( [[0.0 for x in range(2)] for y in range(nsbc)] )
 for s in arange(nsbc): acyc[s,:] = double(tmp[s+i+7].split())

 i = 0
 while '# ratio a_max/a_min' not in lines[i]: i = i+1
 sbratio =  float(lines[i].split()[0])

 # get wave grid and define final cloudy arrays
 i = 0
 while 'abs_cs' not in lines[i]: i = i+1; i1 = i+2
 while 'sct_cs' not in lines[i]: i = i+1; i2 = i+2
 while '(1-g)' not in lines[i]: i = i+1; i3 = i+2
 xw = double( [0.0 for x in range(nw)] )
 a_cs = double( [[0.0 for x in range(nsb)] for y in range(nw)] ); a_cs0 = double( [[0.0 for x in range(nsb)] for y in range(nw)] )
 s_cs = double( [[0.0 for x in range(nsb)] for y in range(nw)] ); s_cs0 = double( [[0.0 for x in range(nsb)] for y in range(nw)] )
 g = double( [[0.0 for x in range(nsb)] for y in range(nw)] ); g_0 = double( [[0.0 for x in range(nsb)] for y in range(nw)] )
 slowp = double( [[0.0 for x in range(nsb)] for y in range(2)] )
 for w in arange(nw):
     xw[w] = double(tmp[i1+w].split()[0])
     a_cs0[w,:] = double(tmp[i1+w].split()[1:])
     s_cs0[w,:] = double(tmp[i2+w].split()[1:])
     g_0[w,:] = double(tmp[i3+w].split()[1:])
 print(' nw = '+str(nw)+' waves found ['+str(xw[0])+','+str(xw[nw-1])+'] Rydberg')

 
#
# read DustEM files
#
 fl = dm[0]+'LAMBDA.DAT'
 print('Reading '+fl)
 f = open(fl,'r')
 dwlines = f.readlines()
 f.close()
 ix=[]
 for i in range(size(dwlines)):  # to skip comments
     if dwlines[i][0] != '#': ix.append(i)
 dwlines = array(dwlines[ix[0]:])
 nwd = int(dwlines[0].split()[0])  # nr of waves
 dwlines = dwlines[1:]
 xwd = double(arange(nwd))
 for i in range(nwd): xwd[i] = dwlines[i].split()[0]
 print(' '+str(nwd)+' waves ['+str(xwd[0])+','+str(xwd[nwd-1])+'] microns')
 xwd = 1e6*2*pi*hbc/xwd/1e-4/1e13/ryd   # convert dustem waves in microns to Ryd
 
 fl = dm[0]+'Q_'+dm[1]+'.DAT'
 print('Reading '+fl)
 f = open(fl,'r')
 dqlines = f.readlines()
 f.close()
 ix=[]
 for i in range(size(dqlines)):  # to skip comments
     if dqlines[i][0] != '#': ix.append(i)
 dqlines = array(dqlines[ix[0]:])
 nad = int(dqlines[0].split()[0])  # nr of sizes

 ad = double(arange(nad))
 qad = double( [[0.0 for x in range(nad)] for y in range(nwd)] )
 qsd = double( [[0.0 for x in range(nad)] for y in range(nwd)] )

 # get size grid, Q and G
 dqlines = dqlines[1:]
 ad = double(dqlines[0].split())*1e-4  # in cm
 print(' '+str(nad)+' size ['+str(ad[0])+','+str(ad[nad-1])+'] cm')
 rho = arange(0)
 if '#' not in dqlines[1].split():
     rho = double(dqlines[1].split())
     print(' rho wrt. size ['+str(rho[0])+','+str(rho[nad-1])+'] cm-3')
 i = 0
 while 'QABS' not in dqlines[i]: i = i+1
 iq = i + 1
 for i in range(nwd):
     qad[i,:] = double( dqlines[iq+i].split() )
 i = 0
 while 'QSCA' not in dqlines[i]: i = i+1
 iq = i + 1
 for i in range(nwd): qsd[i,:] = double( dqlines[iq+i].split() )
 
 fl = dm[0]+'G_'+dm[1]+'.DAT'
 print('Reading '+fl)
 f = open(fl,'r')
 dglines = f.readlines()
 f.close()
 ix=[]
 for i in range(size(dglines)):  # to skip comments
     if dglines[i][0] != '#': ix.append(i)
 dglines = array(dglines[ix[0]:])
 nck = int(dglines[0].split()[0])
 gd = double( [[0.0 for x in range(nad)] for y in range(nwd)] )
 tmp = double(dglines[1].split())*1e-4
 print(' '+str(nck)+' size ['+str(tmp[0])+','+str(tmp[nck-1])+'] cm')
 if nck != nad:
     print('(F) Q and G files nr. of size bins are different')
     return 0
 i = 0
 while 'g-factor' not in dglines[i]: i = i+1
 iq = i + 1
 for i in range(nwd): gd[i,:] = double( dglines[iq+i].split() )

     
#
# interpolate dustem Q's on cloudy wave grid xw for each dustem size
#
 if not (acy[:,0].min() >= ad.min() ) and (acy[:,0].max() <= ad.max() ) :
     print('(F): cloudy size range must fit into that of DustEM')
     return 0
 
# revert dustem grid to get ascending photon energies
 xwd = xwd[::-1]
 for i in range(nad):
     qad[:,i]=qad[::-1,i]; qsd[:,i]=qsd[::-1,i]; gd[:,i]=gd[::-1,i]

     
#
# define cloudy arrays for wave interpolation
#
 a_tmp = double( [[0.0 for x in range(nad)] for y in range(nw)] )
 s_tmp = double( [[0.0 for x in range(nad)] for y in range(nw)] )
 g_tmp = double( [[0.0 for x in range(nad)] for y in range(nw)] )
 
# find bounds for common photon energy range and size range    
 ib = [i for i, s in enumerate(xw) if (s >= min(xwd) and s <= max(xwd)) ]; cb = size(ib)
 iba = [i for i, s in enumerate(ad) if (s > min(acy[:,0]) and s < max(acy[:,0])) ]; cba = size(iba)
 iba.append(min(iba)-1); iba.append(max(iba)+1); iba.sort()  # force inclusion of ad within acy (for plots)

# rebin common photon energy part on cloudy grid 
 for ia in range(nad):
     a_tmp[ib,ia] = interp( xw[ib], xwd,qad[:,ia] )
     s_tmp[ib,ia] = interp( xw[ib], xwd,qsd[:,ia] )
     g_tmp[ib,ia] = interp( xw[ib], xwd,gd[:,ia] )

# size rebin 
 for i in range(nw):
     a_cs[i,:] = interp( acy[:,0], ad,a_tmp[i,:] )
     s_cs[i,:] = interp( acy[:,0], ad,s_tmp[i,:] )
     g[i,:] = 1. - interp( acy[:,0], ad,g_tmp[i,:] )
     
# low and high energy extrapolation on DustEM wave grid
 wx = 1e6*2*pi*hbc/double(wex)/1e13/ryd; wx.sort()  # wex range in Rydberg
 if (wx[0]<min(xwd)) or (wx[1]>max(xwd)):
     print('(F): WEX not within DustEM wave grid')
     return 0
 ibx = [i for i, s in enumerate(xw) if (s >= min(wx) and s <= max(wx)) ]; ibx = array(ibx)
 print('Low energy extrapolation on ',size(ibx),' points and from ',str('{0:.2e}'.format(wx[0])),' to ',str('{0:.2e}'.format(wx[1])),' Rydberg')
 ib = [i for i, s in enumerate(xw) if (s >= min(xwd) and s <= max(xwd)) ]; cb = size(ib)

 for ia in range(nsb):
# Qabs     
     slp = median( log(a_cs[ibx+1,ia]/a_cs[ibx,ia]) / log(xw[ibx+1]/xw[ibx]) )                # low energy: power law from wex range...
     slowp[0,ia] = slp
     a_cs[0:ib[0],ia] = ( xw[0:ib[0]]/xw[ib[0]] )**slp * a_cs[ib[0],ia]
     a_cs[ib[cb-1]:nw,ia] = (a_cs[ib[cb-1],ia]/a_cs0[ib[cb-1],ia]) * a_cs0[ib[cb-1]:nw,ia]    # high energy: use data from *.opc file...
# Qsca (same as Qabs)  
     slp = median( log(s_cs[ibx+1,ia]/s_cs[ibx,ia]) / log(xw[ibx+1]/xw[ibx]) )                
     slowp[1,ia] = slp
     s_cs[0:ib[0],ia] = ( xw[0:ib[0]]/xw[ib[0]] )**slp * s_cs[ib[0],ia]                      
     s_cs[ib[cb-1]:nw,ia] = (s_cs[ib[cb-1],ia]/s_cs0[ib[cb-1],ia]) * s_cs0[ib[cb-1]:nw,ia]
# g     
     g[0:ib[0],ia] = (g[ib[0],ia]/g_0[ib[0],ia]) * g_0[0:ib[0],ia]                            # low and high energy: use data from *.opc file
     g[ib[cb-1]:nw,ia] = (g[ib[cb-1],ia]/g_0[ib[cb-1],ia]) * g_0[ib[cb-1]:nw,ia]
     for i in range(nw): g[i,ia] = min( g[i,ia], 1.)   # impose max(g) = 1 
       
# check plots
 if (chk == 1) or (chk == 2):    
     wnum = 0; fig = plt.figure(wnum,figsize=(7,4))
     fig.canvas.set_window_title('rebinned Qabs  #'+str(wnum))
     plt.suptitle(tit)
     xtit = 'Energy (Ryd)'
     if size(xl) !=2: xr = [xw.min(), xw.max()]
     else: xr = xl; xr.sort()
     ytit = '$Q_{abs}$'
     if size(yl) !=2: yr = [a_cs.min()/2, a_cs.max()*2]
     else: yr = yl; yr.sort()
     plt.xscale('log'); plt.xlabel(xtit)
     plt.yscale('log'); plt.ylabel(ytit)
     plt.xlim(xr); plt.ylim(yr)

     print('Qabs, Qsca and G: plotted sizes (microns)')
     sarr = []
     for i in range(nsb):
         if i%smod == 0:
             plt.plot(xw,a_cs[:,i])
             sarr.append(str(' '+'{0:.1e}'.format(acy[i,0]/1e-4))+' ')
     print(*sarr)
     if chk == 2:
         for i in range(cba): plt.plot(xwd,qad[:,iba[i]],'.')
     
     wnum = 1; fig = plt.figure(wnum,figsize=(7,4))
     fig.canvas.set_window_title('rebinned Qsca  #'+str(wnum))
     plt.suptitle(tit)
     xtit = 'Energy (Ryd)'
     ytit = '$Q_{sca}$'
     if size(yl) !=2: yr = [s_cs.min()/2, s_cs.max()*2]
     else: yr = yl; yr.sort()
     plt.xscale('log'); plt.xlabel(xtit)
     plt.yscale('log'); plt.ylabel(ytit)
     plt.xlim(xr); plt.ylim(yr)

     for i in range(nsb):
        if i%smod == 0:
            plt.plot(xw,s_cs[:,i])
     if chk == 2:
         for i in range(cba): plt.plot(xwd,qsd[:,iba[i]],'.')

     wnum = 2; fig = plt.figure(wnum,figsize=(7,4))
     fig.canvas.set_window_title('rebinned g  #'+str(wnum))
     plt.suptitle(tit)
     xtit = 'Energy (Ryd)'
     ytit = '$g$ = < cos $\\theta$ >'
     if size(yl) !=2: yr = [1e-16, g.max()*2]
     else: yr = yl; yr.sort()
     plt.xscale('log'); plt.xlabel(xtit)
     plt.yscale('log'); plt.ylabel(ytit)
     plt.xlim(xr); plt.ylim(yr)

     for i in range(nsb):
        if i%smod == 0:
            plt.plot(xw,1-g[:,i])
     if chk == 2:
         for i in range(cba): plt.plot(xwd,gd[:,iba[i]],'.')

             
#
# get quantities per H (with size dist acy[:,3]/acy[:,0])
#
 alp = double( [0.0 for x in range(nw)] ) 
 for i in range(nw):  # size rebin per H
     a_cs[i,:] = a_cs[i,:] * acy[:,4]/4.0
     s_cs[i,:] = s_cs[i,:] * acy[:,4]/4.0
     alp[i] = ((a_cs[i,:]+s_cs[i,:]) * acy[:,4]/acy[:,5]).sum()   # inv attenuation length alp = < 3*Qext/4/a > average over grain nr size dist
 alp = alp / acy[:,4].sum()

 
#
# write Qabs, Qsca, g and alp in output file
#
 for i in range(nw):   # write abs. cross-section per H per size bin
     sarr=[str('{0:.6e}'.format(xw[i]))]
     for j in range(nsb): sarr.append(str('{0:.6e}'.format(a_cs[i,j])))             
     sarr = ' '.join(map(str, sarr)); sarr = sarr + '\n'
     fo.write(sarr)

 for i in range(nhl): fo.write(str(hdr[0][i]))
 for i in range(nw):   # write sca. cross-section per H per size bin
     sarr=[str('{0:.6e}'.format(xw[i]))]
     for j in range(nsb): sarr.append(str('{0:.6e}'.format(s_cs[i,j])))             
     sarr = ' '.join(map(str, sarr)); sarr = sarr + '\n'
     fo.write(sarr)

 for i in range(nhl): fo.write(str(hdr[1][i]))
 for i in range(nw):   # write (1-g) per size bin
     sarr=[str('{0:.6e}'.format(xw[i]))]
     for j in range(nsb): sarr.append(str('{0:.6e}'.format(g[i,j])))             
     sarr = ' '.join(map(str, sarr)); sarr = sarr + '\n'
     fo.write(sarr)

 for i in range(nhl): fo.write(str(hdr[2][i]))
 for i in range(nw):   # write attenuation length
     sarr=[str('{0:.6e}'.format(xw[i]))]
     sarr.append(str('{0:.6e}'.format(alp[i])))
     sarr = ' '.join(map(str, sarr)); sarr = sarr + '\n'
     fo.write(sarr)
         
 if show != 0:      
     wnum = 3; fig = plt.figure(wnum,figsize=(7,4))
     fig.canvas.set_window_title('rebinned Qabs per H  #'+str(wnum))
     plt.suptitle(tit)
     xtit = 'Energy (Ryd)'
     if size(xl) !=2: xr = [xw.min(), xw.max()]
     else: xr = xl; xr.sort()
     ytit = '$Q_{abs}$'
     if size(yl) !=2: yr = [a_cs.min()/2, a_cs.max()*2]
     else: yr = yl; yr.sort()
     plt.xscale('log'); plt.xlabel(xtit)
     plt.yscale('log'); plt.ylabel(ytit)
     plt.xlim(xr); plt.ylim(yr)
     print('Displayed Size dist.: first sizes in cm, then grain number per H (nH^-1 * [dn/da]*da)')
     s_arr = []; sd_arr = []
     for i in range(nsb):
         if i%smod == 0:
             plt.plot(xw,a_cs[:,i])
             s_arr.append(str(' '+'{0:.1e}'.format(acy[i,0]))+' ')
             sd_arr.append(str(' '+'{0:.1e}'.format(acy[i,3]/acy[i,0]))+' ')
     print(*s_arr); print(*sd_arr)
     
     wnum = 4; fig = plt.figure(wnum,figsize=(7,4))
     fig.canvas.set_window_title('rebinned Qsca per H  #'+str(wnum))
     plt.suptitle(tit)
     xtit = 'Energy (Ryd)'
     ytit = '$Q_{sca}$'
     if size(yl) !=2: yr = [s_cs.min()/2, s_cs.max()*2]
     else: yr = yl; yr.sort()
     plt.xscale('log'); plt.xlabel(xtit)
     plt.yscale('log'); plt.ylabel(ytit)
     plt.xlim(xr); plt.ylim(yr)
     for i in range(nsb):
        if i%smod == 0:
            plt.plot(xw,s_cs[:,i])

     wnum = 5; fig = plt.figure(wnum,figsize=(7,4))
     fig.canvas.set_window_title('Inverse attenuation length  #'+str(wnum))
     plt.suptitle(tit)
     xtit = 'Energy (Ryd)'
     ytit = '$\\alpha$'
     if size(yl) !=2: yr = [alp.min()/2.,alp.max()*2]
     else: yr = yl; yr.sort()
     plt.xscale('log'); plt.xlabel(xtit)
     plt.yscale('log'); plt.ylabel(ytit)
     plt.xlim(xr); plt.ylim(yr)
     plt.plot(xw,alp)

     wnum = 6; fig = plt.figure(wnum,figsize=(6,5))
     fig.canvas.set_window_title('Dust mass distribution  #'+str(wnum))
     plt.suptitle(tit)
     xtit = 'radius (nm)'
     ytit = '$n_H^{-1}\; a^4\; {\\rm d}n/{\\rm d}a \quad (10^{-29}\; {\\rm cm}^3/{\\rm H})$'
     xr = [0.1,1e4]; yr = [1e-3,1e3]
     plt.xscale('log'); plt.xlabel(xtit)
     plt.yscale('log'); plt.ylabel(ytit)
     plt.xlim(xr); plt.ylim(yr)
     plt.plot(1e7*acy[:,0],1e29*acy[:,5]*(sbratio+1.)*3./2./(sbratio-1)/4./pi,'.')
     plt.plot(1e3*acyc[:,0],1e29*acyc[:,1],'--')
     
 if show != 0:
      plt.show()
      plt.close()

 print('Writing data in '+files[1])
 fo.close() # close output file
 
 return {'acy':acy,'xw':xw,'a_cs':a_cs,'s_cs':s_cs,'g':g,'slowp':slowp}
