
from numpy import *

def plaw(x,par):
# generates a volume normalized power law x^par(0)
# returns distribution in nr of grains : dn/da
# x : grain size array 
# par[0] : power law index
# par[1] : VOLUME normalization 
# par[2] : curvature parameter beta 
# par[3] : large size threshold At

    np = size(x)
    y = x**par[0]

# curvature term
    if ((par[2] != 0) and (par[3] != 0)):
        psgn = par[2]/abs(par[2])
        y = y * ( 1. + abs(par[2])*x/par[3] )**psgn

    vy = x**(4.0) * y 
    dx = log(x[1:np]) - log(x[0:np-1]) 
    yi = sum( (vy[1:np] + vy[0:np-1])*0.5*dx )
    if yi == 0. or isnan(yi) or isinf(yi):
        print('(W) PLAW: unnormalized component par = ',par)
    else:
        y = par[1] * y / yi

    return y


def logn(x,par):
# generates a volume normalized log-normal law 
# returns distribution in nr of grains : dn/da
# x : grain size 
# par[0] : centroid of log-normal
# par[1] : sigma of log-normal
# par[2] : VOLUME normalization
#
# NB: in log grid: dn/dlog x ~ logn(x) or dn/dx ~ logn(x)/x
# with logn(x) = exp( -0.5*( log(x/x0)/sigma )**2 )

    np = size(x)
    x0 = par[0]
    sigma = par[1]
    y = exp(- 0.5 * ( log(x/x0) / sigma )**2 ) / x
    xm = par[0] * exp( 3.*par[1]**2 ) 
    print('(W) LOGN: position of max(a**4 dn/da) @',xm*1e7,' nm')
    
    vy = x**4 * y
    dx = log(x[1:np]) - log(x[0:np-1]) 
    yi = sum( (vy[1:np] + vy[0:np-1])*0.5*dx )
    if yi == 0. or isnan(yi) or isinf(yi):
        print('(W) LOGN: unnormalized component par = ',par)
    else:
        y = par[2] * y / yi
        
    return y


def cut_off(x,par):
# generates the large size cut-off 
# from Weingartner & Draine 2001
# x : grain size 
# par(0) : threshold for cut-off (At)
# par(1) : shape As (roughly the size where cut_off=0.5)

    y = array([1.0 for i in range(size(x))])
    ix = where( x > par[0] )[0]; cnt = size(ix)
    if cnt > 0:
        y[ix] = exp( -( (x[ix]-par[0]) / par[1])**3. )
    else: print('(W) CUT_OFF: no size larger than at =',par[0],' found')

    return y


def create_vdist( ar=[], ns=20, ag=[], rho=0, slaw='PLAW', par=[], cutof=[1.07e-2,0.428], norm=0., fname='', wd=0, plot=1 ):
    
# Doc
    if size(ar) != 2 and size(ag) == 0:
        print('------------------------------------------------------------------------------------------------------------')
        print('def create_vdist( ar=[], ns=20, ag=[], rho=0, slaw="PLAW", par=[], cutof=[1.07e-2,0.428], norm=0.,  ')
        print('                  fname="", wd=0, plot=1 )')
        print('------------------------------------------------------------------------------------------------------------')
        print('generates a dust size distribution for DustEM (SIZE_tt.DAT) in volume normalized to norm')
        print('uses power law or log-normal component')
        print()
        print('returns dictionary {"vdist":vdist,"ag":ag,"rho":rho')
        print()
        print(' AR    (I): array(2) size range in cm ')
        print(' NS    (I): nr of sizes for log-grid ag over ar')
        print(' AG  (I/O): array(ns) of sizes in cm, as input (log-grid of constant step) or as output ')
        print(' RHO   (I): mass density (g/cm3) array(ns). If single value, replicated to all sizes ')
        print(' SLAW  (I): array of size dist. law of grains "PLAW" or "LOGN"')
        print('           [Default = "PLAW" with index -3.5]')
        print(' PAR   (I): array(nlaw,npar) parameters in cgs for the size dist law: ')
        print('           [index,integral,beta,At] for PLAW and [center,width,integral] for LOGN')
        print(' CUTOF (I): parameters for large size cut-off in cm')
        print(' NORM  (I): set vdist integral to norm')
        print(' FNAME (I): string array(2), DustEM path and grain type tt for SIZE_tt.DAT')
        print(' WD    (I): WD01 Rv=3.1 dust size dist, wd=1 for carbon and wd=2 for silicates')
        print(' PLOT   (I): plt=1 to plot dust size dist ')
        print()
        print('>>> from create_vdist import *')
        print()
        print('Example 1: a power-law' )
        print('>>> fnm = ["/Users/lverstra/DUSTEM_LOCAL/dustem4.3","tt"]')
        print('>>> par = array( [[1.0 for j in range(4)] for i in range(1)] )')
        print('>>> par[0][0]=-3.21 ; par[0][1]=1. ; par[0][2]=0.3 ; par[0][3]=1.64e-5')
        print('>>> sd = create_vdist( [3e-8,5e-5], rho=3.3, slaw="plaw", par=par, fname=fnm )')
        print()
        print('Example 2: a log-normal + power law')
        print('>>> par = array( [[1.0 for j in range(4)] for i in range(2)] )')
        print('>>> par[0][0]=4.e-8; par[0][1]=0.2; par[0][2]=0.6  # logn i-line order as in plaw')
        print('>>> par[1][0]=-3.50; par[1][1]=0.4; par[0][2]=0.3; par[0][3]=1.64e-5  # plaw')
        print('>>> sd = create_vdist( [3e-8,5e-5], rho=2.25, slaw=["logn","plaw"], par=par )')
        print()
        print(' Created March 2009, L. Verstraete, IAS')
        print(' Ported to python, May 2018, LV')
        print('------------------------------------------------------------------------------------------------------')
        return
    
# inits
    import matplotlib.pyplot as plt
    xmp = 1.67262158e-24   # proton mass
    
    if size(ar) != 2:
        print('(F) CREATE_VDIST: size range not defined')
        return 0

    if sum(rho) == 0 and wd == 0: 
        print('(F) CREATE_VDIST: you must define a grain mass density')
        return 0
    else:
        if size(rho) == 1: rho = [rho for x in range(ns)]

            
    if size(par) == 0:
        par = array( [[1.0 for x in range(4)] for y in range(1)] )
        par[0][0] = -3.5; par[0][1] = 1.0

    if len(slaw) != size(slaw): slaw = [slaw]
    slaw = [x.upper() for x in slaw]
    nlaw = size(slaw)
    for i in range(nlaw): 
        if (slaw[i] != 'PLAW') and (slaw[i] != 'LOGN'):
            print('(F) CREATE_VDIST: undefined law ',slaw[i])
            print('                  only ''PLAW'' and ''LOGN'' ')
            return 0
        
# WD01 for cirrus emission
    if wd == 1:
        rho = [2.24 for x in range(ns)]   # carbon 230 ppm or Vc = 2.07e-27 cm3/H
        vl = 2.07e-27
        slaw = [ 'LOGN', 'LOGN', 'PLAW' ]
        nlaw = size(slaw)
        par = array( [[1.0 for x in range(4)] for y in range(nlaw)] )
        cutof = [ 0.0107*1.e-4, 0.428*1.e-4 ]
        
        par[0][0] = 3.5e-8       # PAH logn centroid
        par[0][1] = 0.4          # width
        par[0][2] = 0.75*0.26*vl # volume 

        par[1][0] = 3.e-7        # graphite logn centroid
        par[1][1] = 0.4          # width
        par[1][2] = 0.25*0.26*vl # volume

        par[2][0] = -2.54        # graphite BG plaw index
        par[2][1] =  0.74*vl     # volume
        par[2][2] = -0.165       # curvature beta
        par[2][3] =  1.07e-6     # large size threshold at

    elif wd == 2:
        rho = [3.5 for x in range(ns)]   # silicates 36.3 ppm Si or Vs = 2.98e-27 cm3/H
        vl = 2.98e-27
        slaw = ['PLAW']
        nlaw = size(slaw)
        par = array( [[1.0 for x in range(4)] for y in range(nlaw)] )
        cutof = [ 0.164*1e-4, 0.100*1e-4 ]

        par[0][0] = -3.21      # BG plaw index
        par[0][1] =  0.579*vl  # volume (1 corresponds to Vs)
        par[0][2] =  0.30      # curvature beta
        par[0][3] =  1.64e-5   # large size threshold at
        
# set rho array
    rho = array(rho)
        
# define size grid
    if size(ag) == 0: # log
        lar = log(ar)
        da = (lar[1]-lar[0]) / (ns-1)
        a = lar[0] + arange(ns)*da 
        a[ns-1] = lar[1]    # roundup error
        ag = exp(a)
    else:             # from input
        ns = size(ag)
        da = median( log(ai[1:ns]) - log(ai[0:ns-1]) ) / (ns-1)
        ag = ai   

    print('(W) CREATE_VDIST: median log step of size grid ',da)
        
# volume distribution
    if size(par) == 0:
        print('(F) CREATE_VDIST: PAR array not defined')
        return 0
 
    vdist = 0.
    for i in range(nlaw):
        if slaw[i] == 'PLAW':            
            vdist = vdist + ag**(4.0) * plaw(ag,par[i][:])
        else:
            vdist = vdist + ag**(4.0) * logn(ag,par[i][:])
            
    if sum(cutof) != 0:
        cutof = array(cutof)
        print('(W) CREATE_VDIST : cut-off applied, size and scale are ',cutof)
        vdist = vdist * cut_off(ag, cutof)
 
# normalize
    fac = sum(vdist[1:ns]+vdist[0:ns-1])*da / 2.0
    if fac == 0. or isnan(fac) or isinf(fac):
        print('(F) CREATE_VDIST: undefined vdist integral ',fac)
        return 0
    else:
        print('(W) CREATE_VDIST: vdist integral ',fac)
        vdist = vdist / fac              
        if norm>0:
            vdist = norm * vdist        
            mprop = (4.*pi/3) * sum(rho[1:ns]*vdist[1:ns]+rho[0:ns-1]*vdist[0:ns-1])*da / 2.0
            mprop = mprop/xmp               
            print('                  mass fraction ',mprop)
            
    if plot == 1:
        plt.clf() #; plt.figure()
        plt.xscale('log'); plt.xlabel('a (nm)')
        plt.yscale('log'); plt.ylabel('$a^4$ d$n$/d$a$')
        plt.ylim(array([1e-4,2.])*max(vdist))
        plt.plot(1e7*ag,vdist)
        plt.show();
        plt.close()

# write in fname
    if size(fname) == 2:
        fnm = fname[0] + '/data/SIZE_'+fname[1]+'.DAT'
        f = open(fnm,'w')
        f.write('# DUSTEM: Size distribution of grain type '+fname[1]+'\n')
        f.write('#\n')
        f.write('# Nbr of size bins\n')
        f.write('# [ a (cm), a^4*dn/da, rho(a) ]\n')
        f.write('#\n')
        f.write('{0:2d}'.format(ns)+'\n')
        for i in range(ns):
            f.write('{0:.6E} {1:.6E} {2:.6E}'.format(ag[i],vdist[i],rho[i])+'\n')
        f.close()

        print('(W) CREATE_VDIST: mass distribution written in ',fnm)
        
    return {'vdist':vdist, 'ag':ag, 'rho':rho}
