
from numpy import *

def habing_field( x, unit='' ):
# computes the Habing interstellar radiation field SED (ISRF)
# in W/m2 (4*pi*nu*I_nu)
# (from Draine & Bertoldi 1996)
# X	(I): X-grid for the ISRF
# UNIT	(I): unit of the ISRF. Choices are 
#	     'W': wave in um [Default]
#	     'F': frequency in cm-1
#	     'E': energy in eV 

    from numpy import double,where

    if len(unit) == 0: unit = 'W'
    else: unit = unit.upper()

    x = double(x)

    if unit == 'W': x3 = 1.e1 * x 
    elif unit == 'F': x3 = 1.e5 / x
    elif unit == 'E': x3 = 12.4 / x

    field = - x3**3 * 4.1667 + x3**2 * 12.5 - x3 * 4.3333 
    field = 1.e-1 * 3.e8 * 1.e-14 * field 
    
    field[where( field < 0.)] = 0.

    return field   # end habing_field


def mathis_field( x, unit='', chk=0 ):
# computes the Mathis interstellar radiation field SED (ISRF)
# in W/m2 (4*pi*nu*I_nu)
# from Mathis et al. 1983, A&A 128, 212
# X	(I): X-grid for the ISRF
# UNIT	(I): unit of the ISRF. Choices are 
#	    'W': wave in um [Default]
#	    'F': frequency in cm-1
#	    'E': energy in eV

    from numpy import pi,where
    from py4dm import blackbody
    
    if len(unit) == 0: unit = 'W'
    else: unit = unit.upper()

    ly_limit = 9.11267101235e-2
    x = double( x )

#
# visible part 
#
    wdil = 4. * array([ 4.e-13, 1.e-13, 1.e-14])	   # Mathis definition 4*pi*wdil*B_nu
    rr = blackbody( x=x, temp=[ 3.e3, 4.e3, 7.5e3 ], wdil=wdil, unit=unit, chk=chk )
    field = pi * x * rr['bb']

#
# UV part
#

# first convert to (lambda / 1e3 AA)
    if unit == 'W': x3 = 1.e1 * x 
    elif unit == 'F': x3 = 1.e5 / x
    elif unit == 'E': x3 = 12.4 / x 

    field[ where(x3 <= 2.46) ] = 0.
    
    il = where( (x3 >= 1e1*ly_limit)*(x3 < 1.11) )[0]
    if len(il) > 0: field[il] = 1.4817e-6 * x3[il]**(4.4172)

    il = where( (x3 >= 1.11)*(x3 < 1.34) )[0] 
    if len(il) > 0: field[il] = 2.0456e-6 * x3[il]

    il = where( (x3 >= 1.34)*(x3 < 2.46) )[0]
    if len(il) > 0: field[il] = 3.3105e-6 * x3[il]**(-0.6678)

    return field   # end mathis_field
    
    
def create_rfield( temp=0., x=[], isrf=1, wdil=1., wcut=0, ref=0, chk=0, fname='', tau={} ):

    if sum(temp) == 0:
        print('------------------------------------------------------------------------------------------------------------')
        print('def create_rfield( temp=0., x=[], isrf=1, wdil=1, wcut=0, ref=0, chk=0, fname=[DM_path,file_name],tau={} )')
        print('------------------------------------------------------------------------------------------------------------')
        print('generates the radiation field for DustEM (ISRF.DAT)')
        print(' in erg/cm2/s/Hz (4*pi*I_nu)')
        print(' rfield = isrf + wdil*pi*bb(temp) ')
        print(' ISRF from Mathis et al. 1983, A&A 128, 212')
        print(' NB: if blackbody is isotropic set wdil to 4 (to get 4*!pi factor)')
        print()
        print(' returns dictionary {"g0":g0,"chi":chi,"x":x,"rfield":rfield}')
        print()
        print(' temp (I): blackbody temperature (can be an array). If < 0 only ISRF.')
        print(' x    (I): X-grid for the ISRF in microns. Default is 200 pts over 0.01-10^5 microns.')
        print('           (including WCUT point). You want to include WCUT for accurate edges.')
        print(' isrf (I): if set to 0 no ISRF is added, 1 is Mathis (default), 2 is Habing')
        print(' wdil (I): blackbody dilution factor (can be an array). Default is 1.')
        print(' fname(I): string array(2), DustEM path and tt for ISRF_tt.DAT')
        print(' tau  (I): dictionary tau{"tfn":path + abs.filename,"tfac":tau_V value} to apply exp(-tau) to rfield')
        print(' wcut (I): for wave < wcut radiation field is 0. Default is Lyman limit, if < 0 no cut')
        print(' ref  (I): only plot the ISRF, 1=Mathis and 2=Habing')
        print(' chk  (I): to get details')
        print(' g0   (O): factor flux(6-13.6eV) wrt. Mathis field ')
        print(' chi  (O): scaling factor at 100nm wrt. Mathis field')
        print()
        print('Examples:')
        print('>>> from create_rfield import *')
        print('>>> path="/Users/lverstra/DUSTEM_LOCAL/dustem4.3"')
        print('>>> fnm=[path,"tt"]')
        print('>>> tt=create_rfield( temp=[2e4,5e4], wdil=[1.e-14,1.e-16], fname=fnm)')
        print('>>> tau={"tfn":path+"/out/EXT_MC10.RES","tfac":2.}')
        print('>>> tt=create_rfield( temp=-1,ref=1,tau=tau )')
        print()
        print(' Created Aug. 2009, L. Verstraete, IAS')
        print(' Force the wcut point, May 2011, LV')
        print(' Ported to python, May 2018, LV')
        print('------------------------------------------------------------------------------------------------------')
        return

# inits
    from py4dm import blackbody
    import matplotlib.pyplot as plt
    
    unit = 'W'
    ly_limit =  9.11267101235e-2
    xi = [ ly_limit, 2.07e-1 ] # for G0 6-13.6 eV in microns
    x1 = 1.e-1 # for CHI

# get x-scale in microns   
    if wcut == 0: wcut=ly_limit  
    if len(x) == 0:
        nx = 199
        xb = [ 1.e-2, 1.e5 ] # wave boundaries in microns
        dx = (log(xb[1])-log(xb[0])) / (nx-1)
        x = log(xb[0]) + arange(nx)*dx
        x = exp(x)
        if wcut > 0: x = append(x,wcut)    # add wcut to avoid coarse edge
        x = unique(x); x = sort(x)
        nx = size(x)
        if x[nx-1] != xb[1]: x[nx-1]=xb[1]  # check rounding
    else:
        print('(W) CREATE-RFIELD : using input wave x')
        x = double(array(x))
        nx = size(x)
        ix = where( abs(x-wcut)/x <= 0.01)[0]
        if len(ix) == 0:
            print('(W) CREATE_RFIELD: your x-grid does not contain wcut.')
            print('                   Should be included if radiation is 0 below wcut.')
            
#
# get intensity            
#
    rfield = 0.

# get Mathis isrf for normalization
    rmathis = mathis_field( x,unit=unit, chk=chk )
    rmathis =  1.e3*x*rmathis/3.e14 # erg/cm2/s/Hz
    
    if isrf == 1:
        print('(W) CREATE_RFIELD : Mathis ISRF')
        rfield = rmathis
    elif isrf == 2: 
        print('(W) CREATE_RFIELD : Habing ISRF')
        rhabing = habing_field(x,unit=unit)
        rhabing =  1.e3*x*rhabing/3.e14 # erg/cm2/s/Hz
        rfield = rhabing

# get blackbody  
    ntemp = size(temp)
    if sum(temp) > 0:
        tt = blackbody( x=x, temp=temp, unit=unit, wdil=wdil, chk=chk )
        bb = tt['bb'] * 1.e3 * pi * x**2 / 3.e14   # erg/cm2/s/Hz
        print('(W) CREATE_RFIELD : adding BB with T= ',temp)
        print('                dilution factor wdil= ',wdil)
    else:
        print('(W) CREATE_RFIELD : no BB added')
        bb = 0.0      
    rfield = rfield + bb

# apply cut
    if wcut > 0:
        ix = where(x < wcut)[0]; cx = size(ix)
        if cx > 0: rfield[ix] = 0.

# apply tau
    if len(tau) > 0:
        f = open(tau['tfn'],'r')
        lines = f.readlines()
        f.close()
        print( '(W) CREATE_RFIELD: applying exp(-tau) with tau_V='+str('{0:2s}{1:.4e}'.format('', tau['tfac']))+' from '+tau['tfn']) 
        ix=[]
        for i in range(size(lines)):  # to skip comments
            if lines[i][0] != '#': ix.append(i)
        print(lines[ix[0]-1],end='')
        lines = lines[ix[0]:]
        ntype = int(lines[0].split()[0]) # nr of grain types
        nlamb = int(lines[0].split()[1])    # nr of wavelengths
        lines = lines[1:]
        xlamb = double( [0.0 for x in range(nlamb)] )
        stmp = double( [0.0 for x in range(nlamb)] )
        for i in range(nlamb):
            xlamb[i] = double(lines[i].split()[0])
            stmp[i] = double(lines[i].split()[2*ntype+1])
        tt = list(abs(xlamb-0.55)); iv = tt.index(min(tt))
        yv = tau['tfac'] / stmp[iv]
        sext = interp(x,xlamb,stmp)*yv
        rfield = rfield * exp(-sext)
            
# get G0
    ig = where( (x >= xi[0])*(x <= xi[1]) )[0]; cg = size(ig)
    if cg > 0:
        xx=x[ig]; yy=rfield[ig]; rr=rmathis[ig]
        s1 = sum( (yy[1:cg]+yy[0:cg-1])*(xx[1:cg]-xx[0:cg-1])/2. )
        s2 = sum( (rr[1:cg]+rr[0:cg-1])*(xx[1:cg]-xx[0:cg-1])/2. )
        g0 = s1/s2
    else: g0 = 0.
    print(' G0 =', str('{0:2s}{1:.4e}'.format('', g0)))

# get chi
    ig = where( abs(2.*(x-x1)/(x+x1)) <= 0.1)[0]; cg = size(ig) 
    if cg > 0:
        ix = where( abs(x[ig]-x1) == min(abs(x[ig]-x1)) )[0]
        ix = ix[0]
        chi = rfield[ig[ix]] / rmathis[ig[ix]]  
    else: chi = 0.
    print(' chi=', str('{0:2s}{1:.4e}'.format('',chi)) )
        
# plot
    plt.clf()
    #plt.figure()
    plt.xscale('log'); plt.xlabel('$\lambda$ (microns)')
    stt = '4$\pi\\nu$ $I_{\\nu}$ (erg/cm$^2$/s/Hz)'
    plt.yscale('log'); plt.ylabel(stt)
    plt.plot(x,3e14*rfield/x,color='red')
    if ref == 1:
        if isrf == 1: plt.plot(x,3e14*rmathis/x,color='blue')
        if isrf == 2: plt.plot(x,3e14*rhabing/x,color='blue')
    plt.show()

# write file
    if size(fname) == 2:
        fnm = fname[0] + '/ISRF_'+fname[1]+'.DAT'
        f = open(fnm,'w')
        f.write('# DUSTEM: exciting radiation field featuring\n')
        if int(isrf) == 1: f.write('# Mathis ISRF\n')
        if int(isrf) == 2: f.write('# Habing ISRF\n')
            
        if sum(temp) > 0:
            d = str(type(temp))
            if d.find('list') == -1: temp = [temp]
            d = str(type(wdil))
            if d.find('list') == -1: wdil = [wdil]
            a =  '# Blackbody with     T='
            a1 = '# dilution factor wdil='  
            for i in range(ntemp):
                a =  a    + str('{0:2s}{1:.6e}'.format('', temp[i]) )
                a1 = a1   + str('{0:2s}{1:.6e}'.format('', wdil[i]) )
            f.write(a+'\n'); f.write(a1+'\n')
            
        if len(tau) > 0:
            f.write('# applied exp(-tau) with tau_V= '+str('{0:2s}{1:.4e}'.format('', tau['tfac']))+' from '+tau['tfn']+'\n')
            
        f.write('# G0='+str('{0:2s}{1:.4e}'.format('', g0))+', chi='+str('{0:2s}{1:.4e}'.format('', chi))+'\n')
        f.write('# Nbr of points\n')
        f.write('# wave (microns), 4*pi*Inu (erg/cm2/s/Hz)\n')
        f.write('{0:4d}'.format(nx)+'\n')
        for i in range(nx):
            f.write('{0:.6e}{1:2s}{2:.6e}{3:2s}'.format(x[i],'',rfield[i],'')+'\n')
        f.close()
        print('(W) CREATE_RFIELD: radiation field written in ', fnm )

    return { 'g0':g0, 'chi':chi, 'x':x, 'rfield':rfield }
