def band_cc( w0,xs,ys,xt,yt ):

    from numpy import double,interp,median,size,unique,where
    
    if w0 == 0.:
        print('-----------------------------------------------------------------------------------------------')
        print('def band_cc( w0,xs,ys,xt,yt ):')
        print('-----------------------------------------------------------------------------------------------')
        print(' returns tuple sed,y0,rr,cc,xi,filt (see below)')
        print('')
        print(' w0    (I): wave centroid of filter (any unit)')
        print(' xs    (I): input wave-grid for SED (same unit as W0)')
        print(' ys    (I): input SED in erg/cm2/sr')
        print(' xt    (I): wave-grid of band transmission (same unit as W0)')
        print(' yt    (I): transmission values (arb. unit)')
        print(' sed   (O): flux in given instrument band (erg/cm2/sr) assuming a photomultiplier with ')
        print('            nuInu = const as reference. Otherwise adapt input yt and/or ys in <get_band_flux>')
        print(' y0    (O): SED at band center')
        print(' rr    (O): undersampling ratio: dx(SED)/dx(filt), usually 1. ')
        print('            If rr<1 filter oversampled in flux integration')
        print(' cc    (O): color correction')
        print(' xi    (O): x-grid used for band integration (ordered same unit as W0')
        print(' filt  (O): transmission used for band integration')        
        return

    vlow = 1e-6  # lowest transmission
        
# prepare filter array
    xtt = xt; ytt = yt; ytt = ytt/max(ytt)
    ix = unique(xtt,return_index=True); xtt = ix[0]
    ytt = ytt[ix[1]]  # unique also sorts
    ixn = where( (xtt>0.) * (ytt>vlow) )[0]; cn = size(ixn)
    if cn == 0:  # if no good point in filter, band flux and the rest is 0 
        sed = 0.; rr = 0.; y0 = 0.; cc = 0.; nint = size(xt)
        return sed
    else: xtt = xtt[ixn]; ytt = ytt[ixn]
    nfilt = size(xtt)
    
# prepare SED array
    ix = unique(xs,return_index=True); xss = ix[0]
    yss = ys[ix[1]]
    ix = where( (xss >= min(xtt)) * (xss <= max(xtt)) )[0]; cx = size(ix)
    if cx > 0:  # add points at edges for interpolation
        i1 = 0 
        if ix[0] > i1: i1 = ix[0]-1
        i2 = size(xss)-1
        if ix[cx-1] < i2: i2 = ix[cx-1]+1
        xss = xss[i1:i2+1]
        yss = yss[i1:i2+1]
    else:   # if no good point in SED, band flux and the rest is 0 
        sed=0.; rr=0.; y0=0.; cc=0.; nint = size(xt)
        return sed
    nsed = size(xss)

# get filter sampling f_res = dx(filt)
    ix = where( ytt/max(ytt) > vlow)[0]; cx = size(ix)
    f_res = abs( median( xtt[ix[1:cx]] - xtt[ix[0:cx-1]] ) )

# Interpolate to compute integration
    if nfilt >= nsed:  # finer filter grid: integrate band flux on filter x-grid
        xi = xtt  
        rr = 1.   # SED sampling is filter sampling
        yi = interp(xi,xss,yss)
        filt = ytt
    else:              # finer SED grid: integrate band flux on SED x-grid
        xi = xss; yi = yss 
        s_res = abs( median( xi[1:nsed]-xi[0:nsed-1] ) ) # get SED sampling: s_res = dx(SED) and sampling ratio
        rr = s_res / f_res
        filt = interp(xi, xtt, ytt)
        io = where( (xi > max(xtt)) + (xi < min(xtt)))[0]; co = size(io)
        if co > 0: filt[io] = 0.
        ixn = where( filt < 0.)[0]; cn = size(ixn) 
        if cn > 0: filt[ixn] = 0.

# Integrate on filter bandpass (log-grid), trapezium safe and accurate enough
    np = size(xi)
    nint = np
    dx = xi[1:np] - xi[0:np-1]
    yint = yi*filt
#  a1 = trapz(yint,x=xi)
    yint = (yint[0:np-1] + yint[1:np]) / 2.
    a1 = sum( yint*dx ) 
    yint = filt
#  a2 = trapz(yint,x=xi)
    yint = (yint[0:np-1] + yint[1:np]) / 2.
    a2 = sum( yint*dx )
  
# get color correction, color corrected band flux and sed
    y0 = interp( w0, xi,yi )
    cc = a1/a2/y0
    sed = y0*cc
    return sed,y0,cc,rr,xi,filt  # band_cc


def get_band_flux( path, inst='', xs=0., ys=0., smi={0}, mtag='emis', unit='', verb=0 ):

  if inst == 0:
     print('-----------------------------------------------------------------------------------------------------------------------------')
     print('def get_band_flux( path, inst='', xs=0., ys=0., smi={0}, unit='', verb=0 )')
     print('-----------------------------------------------------------------------------------------------------------------------------')
     print(' returns new dictionary SMDAT with band or spectral flux (MJy/sr) in instrument filter.')
     print(' or fills band flux in input structure SMI containing M_EMIS field')
     print(' Band flux integration always done in lambda, Deviations less than 1% for transmissions given vs. frequency.')
     print('')
     print(' SMDAT = { "M_EMIS": { "UNIT": for SED (erg/s/cm2/sr),              # model sub-dictionary among "EMIS", "EXT" and "POL" ')
     print('                       "X":    xgrid for SED (microns), ')
     print('                       "Y":    input SED, ')
     print('                       "YP":   input polarized SED           }, ')
     print('           "COM"   : string for doc,')
     print('           "NPAR"  : nr of parameters for overall CHI2,')
     print('           "CHI2"  : chi-square on all data,')
     print('           "I_INST": { "NAME": string array of band names,          # instrument sub-dict., INST among list below ')
     print('                        "X":   band center in microns, ')              
     print('                       "YD":   data points (erg/s/cm2/sr),')
     print('                       "ERR":  error on YD same unit,')
     print('                       "YM":   SED of model in bands (erg/s/cm2/sr), ')
     print('                       "FLX":  flux of model in bands (MJy/sr),     # photometric instrument only ')
     print('                       "CC":   color correction in bands,           #     "           "        "  ')
     print('                       "RR":   band sampling ratio flux/filt,       #     "           "        "  ')                
     print('                       "ISEL": mask for band selection, 0:discard band, 1=take band, ')
     print('                       "UNIT": info on units for X,YD,YM,FLX, ')
     print('                       "CHI2": chi-square on current bands masked with ISEL, ')
     print('                       "TRANS": { "X": grid for band transmission,  # on query with NTRANS')
     print('                                  "Y": band transmission }   }, } ')
     print('')
     print(' PATH (I): path for FILTERS directory')
     print(' INST (I): array of strings for band set, so far CAM, DIRBE, HFI, IRAC, IRAS, LFI, MIPS, MIRIM, NIRCAM, PACS, SPIRE, WMAP')
     print(' XS   (I): SMDAT creation, input wave array in microns ')
     print(' YS   (I): SMDAT creation, input SED in erg/s/cm2/sr, array(nwave,ntype) allows for several SEDs (grain types or models)')
     print(' SMI  (I): input SMDAT structure, *must at least contain M_EMIS field*. If set GET_BAND_FLUX will return filled SMI')
     print(' MTAG (I): tag for model, "emis" or "emispol" ')
     print('')
     print(' Dependencies: ')
     print('  BAND_CC, ADD_MOD [STR_TOOLS], HFI_READ_BANDPASS, GET_HFIBOLO_LIST, HFI_COLOR_CORRECTION ')
     print('')
     print(' Examples: ')
     print(' >>> from band_flux import *')
     print(' >>> path = "~/dustem4.3/py/"')
     print(' SMDAT creation: >>> sm = get_band_flux( path, [''iras'',''hfi''], xs=xs, ys=ys)')
     print(' SMDAT filling:  >>> sm = get_band_flux( path, [''spire''],smi=smi)               -> uses xs and ys from SMI (M_EMIS.X and Y)')
     print(' SMDAT filling:  >>> sm = get_band_flux( path, [''spire''],xs=xs,ys=ys,smi=smi)   -> force xs and ys in SMI')
     print('')
     print(' created in IDL by Laurent Verstraete, IAS, Oct-dec 2010, Bandpass data from FILTERS')
     print(' ported to Python by LV, spring 2021')
     print('')
     print('-------------------------------------------------------------------------------------------------------------------------')
     return 0

#  init & check
  import glob,re
  from numpy import append,argsort,array,char,double,interp,reshape,shape,size,sort,sqrt,where
  from scipy import io
  from astropy.io import fits
  from str_tools import add_mod,str_inst
  from band_flux import band_cc
  from hfi_tuls import hfi_read_bandpass,get_hfibolo_list
  
  clight = 2.9979246e10  # cm/s

  i_list = ['AKARI','CAM','DIRBE','HFI','IRAC','IRAS','LFI','MIPS','MIRI','NIRCAM','PACS','SPIRE','WMAP']    # list of available band instruments
  n_inst = size(inst)
  inst = char.upper(char.strip(inst)) # inst = [x.strip() for x in inst];  inst = [x.upper() for x in inst]
  ix = [i for i, s in enumerate(i_list) if s in inst]; cx = size(ix)
  if cx == 0:
    print('(W) GET_BAND_FLUX: band instrument not found,',inst,' only create "M_EMIS"')
    smi = {}
  nsw = size(xs)  # nr of input wave points
     
# check whether this is a SMDAT creation
  stag = list(smi)
  if size(stag) == 0:
      new_str = 1  # new SMDAT dict.
  else:
      ix = [i for i, s in enumerate(stag) if s == 'M_EMIS']
      if size(ix) == 0:
          new_str = 1  # 'M_EMIS' must be present in SMDAT          
      else: new_str = 0
  
# get start structure
  if new_str == 1:        # SMDAT creation
      if verb == 1: print('(W) GET_BAND_FLUX: creating smdat or adding "M_EMIS" ')
      tt = shape(ys); ntype = tt[0]
      if (nsw+size(shape(ys)) == 0) or (tt[1] != nsw):
          print('(F) GET_BAND_FLUX: creating SMDAT, you must define xs(n2), ys(n1,n2)')
          return -1        
      ix = argsort(xs)      
      xw = xs[ix]; yw = ys[:,ix]
      smdat = { 'COM':'', 'NPAR':0, 'CHI2':0. }
      smdat = add_mod( nsz=[nsw,ntype],sf=smdat,tag=mtag )
  elif new_str == 0:  # filling existing SMDAT 
     smdat = smi
     xw = smdat['M_EMIS']['X']
     yw = smdat['M_EMIS']['Y'] 
     if nsw == size(xw): good = 1 
     if shape(ys) == shape(yw): good = good + 1 
     if good == 2:
         if verb == 1: print('(W) GET_BAND_FLUX: copying input xs and ys in SMI')
         ix = argsort(xs)
         tt = shape(ys)
         if tt[0] > 1:
             ntype = tt[0]
         else: ntype = 1
         xw = xs[ix]; yw = ys[:,ix]
     if size(shape(unit)) == 0: unit = smi['M_EMIS']['UNIT']

  if size(shape(unit)) == 0: unit = 'x(microns) SED(erg/s/cm2/sr)'
  smdat['M_EMIS']['UNIT'] = unit
  smdat['M_EMIS']['X'] = xw
  smdat['M_EMIS']['Y'] = yw
  tag = array(list(smdat))


# instrument loop
  for ixs in range(n_inst):

     if (inst[ixs] == 'AKARI'):
#        bnm = ['N2','N3','N4','S7','S9W','S11','L15','L18W','L24','N60','WS','WL','N160']  # IRC + FIS inst.       
#        wbd = [2.,3.,4.,7.,9.,11.,15.,18.,24.,65.,90.,140.,160.]
        bnm = ['S9W','L18W','N60','WS','WL','N160']  # IRC + FIS inst.       
        wbd = [9.,11.,15.,18.,24.,65.,90.,140.,160.]
        nband = size(wbd)        # nr of bands
        ntrans = 4700            # max nr of points in transmission grid
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_AKARI' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where(ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)'
            dir_band = path + '/AKARI/filter_'
            for k in range(nband):
                fn = dir_band + bnm[k] + '.txt'
                f = open(fn,'r'); lines = f.readlines(); f.close()
                if verb == 1: print(fname)
                nt = size(lines)
                xt = array([double(lines[i].split()[0]) for i in range(nt)]); nt = size(xt)
                yt = array([double(lines[i].split()[1]) for i in range(nt)])
                if wbd[k] < 24.: yt = yt/xt  # from F Galliano note and AKARI doc
                nu0 = 1e4*clight/wbd[k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
                smdat[ttg[itx]]['RR'][k] = rb
        else: # no filter read for existing 'I_AKARI' key
            ttg = tag
            wbd =  smdat[ttg[itx]]['X']
            for k in range(nband):
               nu0 = 1e4*clight/wbd[k]
               xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
               yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
               if wbd[k] < 24.: yt = yt/xt  # from F Galliano note and AKARI doc
               for itp in range(ntype):
                   smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                   smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                   smdat[ttg[itx]]['CC'][itp,k] = tc
               smdat[ttg[itx]]['RR'][k] = rb                
        # end AKARI           
      
     if inst[ixs] == 'CAM':         
        nband_sw = 11
        nband_lw = 10
        nband = nband_sw + nband_lw  # nr of bands
        ntrans = 160                 # max nr of points in transmission grid
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)        
        if (new_str == 1) or (ctg == 0):  # creating 'I_CAM' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            dct = io.readsav( path+'/ISO/cam_filter_transmission.save' )
            wbd = append( dct['sw_lambda_lc'][:nband_sw],dct['lw_lambda_lc'][:nband_lw])
            smdat[ttg[itx]]['X']= array(wbd)
            smdat[ttg[itx]]['NAME'] = wbd.astype(str)
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)'         
            for k in range(nband_sw):
                ind = where(dct('sw_lambda_arr')[k,:] != 0.)[0] # band first, reversed indices
                xt = dct('sw_lambda_arr')[k,ind]
                yt = dct('sw_trans_arr')[k,ind]
                w0 = dct['sw_lambda_lc'][k]
                nu0 = 1e4*clight/w0
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( w0,xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi           
            for k in range(nband_lw):
                ind = where(dct('lw_lambda_arr')[k,:] != 0.)[0]
                xt = dct('lw_lambda_arr')[k,ind]; nt = size(xt)
                yt = dct('lw_trans_arr')[k,ind]
                w0 = dct('lw_lambda_lc')[k]
                nu0 = 1e4*clight/w0                       
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,nband_sw+k],y0,tc,rb,xi,yi =  band_cc( w0,xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,nband_sw+k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,nband_sw+k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][nband_sw+k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,nband_sw+k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,nband_sw+k] = yi
        else: # no filter read for existing 'I_CAM' key
            ttg = tag
            wbd =  smdat[ttg[itx]]['X']
            for k in range(nband):
                nu0 = 1e4*clight/wbd[k]
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb   
        # end CAM
           
     elif (inst[ixs] == 'DIRBE'):     
        nband = 10              # nr of bands
        ntrans = 800            # max nr of points in transmission grid
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_DIRBE' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where(ttg == 'I_'+inst[ixs])[0])
            dir_band = path + '/DIRBE'
            fname = dir_band + '/dirbe_response.asc'
            wav = double([0.0 for i in range(ntrans)])
            trans = double([[0.0 for i in range(nband)] for i in range(ntrans)])
            f = open(fname,'r'); lines = f.readlines(); f.close()
            if verb == 1: print(fname)
            lines = lines[12:] # to skip comments
            for i in range(ntrans):
                wav[i] = double(lines[i].split()[0])
                trans[i,:] = double( lines[i].split()[1:] )
            wbd = double([1.25,2.2,3.5,4.9,12,25.,60.,100.,140.,240.])
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['NAME'] = ['DIRBE'+str(x+1) for x in range(10)]
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)'
            for k in range(nband):
                nu0 = 1e4*clight/wbd[k]
                ix = where(trans[:,k]>0.)[0]
                xt=wav[ix]; yt=trans[ix,k]; nt=size(xt)
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else: # no filter read for existing 'I_DIRBE' key
            ttg = tag
            wbd =  smdat[ttg[itx]]['X']
            for k in range(nband):
                nu0 = 1e4*clight/wbd[k]
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb     
        # end DIRBE
        
     elif (inst[ixs] == 'HFI'):
        nband = 6
        wbd = [ 100.,143.,217.,353.,545.,857.]   # band center in GHz
        bnm = [str(int(wbd[i])) for i in range(nband)]
        wbd = double(wbd)
        hfi_bp_version = 'v101'
        ntrans = 800
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_HFI' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = 1e-5*clight/wbd  # microns
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)' 
            dir_band = path + '/HFI/OFFICIAL/'
            bp = hfi_read_bandpass( version=hfi_bp_version, path_rimo=dir_band )
            nu = 1e-5*clight/xw    # microns to GHz
            for k in range(nband):
                bolo_list = get_hfibolo_list( channel = bnm[k] )
                n_bolo_list = size(bolo_list)
                yt = 0e0
                for idet in range(n_bolo_list): # ----- get transmission ------
                    q_det = [d for d,s in enumerate(bp) if bolo_list[idet] in bp[d]['NAME']][0]
                    trans_d = double(bp[q_det]['TRANS'])
                    q_trans = where( trans_d > 1e-6 )[0]
                    freq_d = double(bp[q_det]['FREQ'][q_trans]) # in GHz
                    trans_d = trans_d[q_trans]
                    if idet == 0: # use 1st freq-grid as reference
                        freq_0 = freq_d
                        n0 = size(freq_0)
                        yt = yt + trans_d
                    else: yt = yt + interp(freq_0,freq_d,trans_d)
                xt = 1e4*clight/freq_0   # microns
                w0 = 1e-5*clight/wbd[k] 
                nu0 = 1e9*wbd[k] # Hz
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( w0,xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else:     # no filter read for existing 'I_HFI' key
            ttg = tag
            for k in range(nband):
                w0 = 1e-5*clight/wbd[k] # microns
                nu0 = 1e4*clight/wbd[k] # Hz
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb                     
       # end HFI
        
     elif (inst[ixs] == 'IRAC'):
        nband = 4
        ntrans = 410
        wbd = [ 3.550, 4.493, 5.731, 7.872 ] # in microns
        bnm = ['IRAC'+str(x+1) for x in range(4)]
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_IRAC' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)'       
            dir_band = path + '/Spitzer/'
            for k in range(nband):
                fn = dir_band + bnm[k].lower() + '.dat'
                f = open(fn,'r'); lines = f.readlines(); f.close()
                if verb == 1: print(fname)
                lines = lines[3:] # to skip comments
                nt = size(lines)
                xt = array([double(lines[i].split()[0]) for i in range(nt)]); nt = size(xt)
                yt = array([double(lines[i].split()[1]) for i in range(nt)])
                nu0 = 1e4*clight/wbd[k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
                smdat[ttg[itx]]['RR'][k] = rb
        else: # no filter read for existing 'I_IRAC' key
            ttg = tag
            wbd =  smdat[ttg[itx]]['X']
            for k in range(nband):
               nu0 = 1e4*clight/wbd[k]
               xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
               yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
               for itp in range(ntype):
                   smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                   smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                   smdat[ttg[itx]]['CC'][itp,k] = tc
               smdat[ttg[itx]]['RR'][k] = rb                
        # end IRAC
        
     elif (inst[ixs] == 'IRAS'):
        nband = 4
        ntrans = 140
        bnm = ['12','25','60','100']
        wbd = double(bnm)
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_IRAS' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)' 
            dir_band = path + '/IRAS/IRAS'
            for k in range(nband):
               fn = dir_band + bnm[k]+'.txt'
               f = open(fn,'r'); lines = f.readlines(); f.close()
               if verb == 1: print(fname)
               lines = lines[2:] # to skip comments
               nt = size(lines)
               xt = array([double(lines[i].split()[0]) for i in range(nt)])
               yt = array([double(lines[i].split()[1]) for i in range(nt)])
               nu0 = 1e4*clight/wbd[k]
               for itp in range(ntype):
                   smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                   smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                   smdat[ttg[itx]]['CC'][itp,k] = tc
               ni = size(xi)
               smdat[ttg[itx]]['RR'][k] = rb
               smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
               smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else: # no filter read for existing 'I_IRAS' key
            ttg = tag
            wbd =  smdat[ttg[itx]]['X']
            for k in range(nband):
               nu0 = 1e4*clight/wbd[k]
               xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
               yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
               for itp in range(ntype):
                   smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                   smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                   smdat[ttg[itx]]['CC'][itp,k] = tc
               smdat[ttg[itx]]['RR'][k] = rb
        # end IRAS
        
     elif (inst[ixs] == 'LFI'):
        nband = 3
        wbd = [ 30.,44.,70. ]   # band center in GHz
        bnm = [str(int(wbd[i])) for i in range(nband)]
        wbd = double(wbd)
        ntrans = 210
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_LFI' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = 1e-5*clight/wbd
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)' 
            dir_band = path + '/LFI/LFI'
            for k in range(nband):
                fn = glob.glob(dir_band+'*'+bnm[k]+'*.dat'); nfiles=size(fn)
                if verb == 1: print(fn)
                for i in range(nfiles):
                    f = open(fn[i],'r'); lines = f.readlines(); f.close()
                    nt = size(lines)
                    if i == 0:
                        nu = [double(lines[i].split()[0]) for i in range(nt)]
                        r0d0 = [double(lines[i].split()[1]) for i in range(nt)]
                        r0d1 = [double(lines[i].split()[2]) for i in range(nt)]
                        r1d0 = [double(lines[i].split()[3]) for i in range(nt)]
                        r1d1 = [double(lines[i].split()[4]) for i in range(nt)]
                    else:
                        nu2 = [double(lines[i].split()[0]) for i in range(nt)]
                        r0d0_2 = [double(lines[i].split()[1]) for i in range(nt)]
                        r0d1_2 = [double(lines[i].split()[2]) for i in range(nt)]
                        r1d0_2 = [double(lines[i].split()[3]) for i in range(nt)]
                        r1d1_2 = [double(lines[i].split()[4]) for i in range(nt)]
                        r0d0 = r0d0 + interp(nu,nu2,r0d0_2)
                        r1d0 = r1d0 + interp(nu,nu2,r1d0_2)
                        r0d1 = r0d1 + interp(nu,nu2,r0d1_2)
                        r1d1 = r1d1 + interp(nu,nu2,r1d1_2)
                    yt = sqrt(double(r0d0)**2+double(r1d1)**2+double(r1d0)**2+double(r0d1)**2)  # to be confirmed 
                    xt = 1e-5*clight/double(nu)
                    w0 = 1e-5*clight/wbd[k] # microns
                    nu0 = 1e9*wbd[k] # Hz
                    for itp in range(ntype):
                        smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( w0,xw,yw[itp,:],xt,yt )
                        smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                        smdat[ttg[itx]]['CC'][itp,k] = tc
                    ni = size(xi)
                    smdat[ttg[itx]]['RR'][k] = rb
                    smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                    smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else:     # no filter read for existing 'I_LFI' key
            ttg = tag
            for k in range(nband):
                w0 = 1e-5*clight/wbd[k] # microns
                nu0 = 1e4*clight/wbd[k] # Hz
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb       
        # end LFI
        
     elif (inst[ixs] == 'MIPS'):
        bnm = [ '24', '70', '160' ]  # band name
        wbd = [ 23.68, 71.42, 155.90 ]  # band center in microns
        nband = size(wbd)
        ntrans = 400
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_MIPS' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)' 
            dir_band = path + '/Spitzer/MIPSfiltsumm_'
            for k in range(nband):
                fn = dir_band + bnm[k]+'.txt'
                f = open(fn,'r'); lines = f.readlines(); f.close()
                if verb == 1: print(fname)
                lines = lines[5:] # to skip comments
                nt = size(lines)
                xt = array([double(lines[i].split()[0]) for i in range(nt)])
                yt = array([double(lines[i].split()[1]) for i in range(nt)])
                yt = yt*wbd[k]**3/xt**4  # from F Galliano note and MIPS doc
                nu0 = 1e4*clight/wbd[k]
                for itp in range(ntype):
                    ytw = yw[itp,:] * (xw/wbd[k])**3  # from F Galliano note and MIPS doc
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,ytw,xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else: # no filter read for existing 'I_MIPS' key
            ttg = tag
            for k in range(nband):
                nu0 = 1e4*clight/wbd[k]
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                yt = yt*wbd[k]**3/xt**4  # from F Galliano note and MIPS doc
                for itp in range(ntype):
                    ytw = yw[itp,:] * (xw/wbd[k])**3  # from F Galliano note and MIPS doc
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,ytw,xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb    
        # end MIPS

     elif (inst[ixs] == 'MIRI'):
        bnm = ['F560W','F770W','F1000W','F1130W','F1280W','F1500W','F1800W','F2100W','F2550W'] # band names
        wbd = [5.6,7.7,10.0,11.3,12.8,15.0,18.0,21.0,25.5] # band center in microns
        nband = size(wbd)
        ntrans = 5002
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_MIRI' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)' 
            dir_band = path + '/MIRI/JWST_MIRI.'
            for k in range(nband):
                fn = dir_band + bnm[k]+'.dat'
                f = open(fn,'r'); lines = f.readlines(); f.close()
                if verb == 1: print(fname)
                nt = size(lines)
                xt = array([double(lines[i].split()[0]) for i in range(nt)])*1e-4  # convert Ang to microns
                yt = array([double(lines[i].split()[1]) for i in range(nt)])
                nu0 = 1e4*clight/wbd[k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else: # no filter read for existing 'I_MIPS' key
            ttg = tag
            for k in range(nband):
                nu0 = 1e4*clight/wbd[k]
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb    
        # end MIRI
                
     elif (inst[ixs] == 'MIRIM'):
        bnm = ['F560W','F770W','F1000W','F1130W','F1280W','F1500W','F1800W','F2100W','F2550W'] # band names
        wbd = [5.6,7.7,10.0,11.3,12.8,15.0,18.0,21.0,25.5] # band center in microns
        nband = size(wbd)
        ntrans = 5002
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_MIRIM' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)' 
            fname = path + '/MIRIM/filters_trans.txt'
            f = open(fname,'r'); lines = f.readlines(); f.close()
            lines = lines[2:] # to skip comments
            nt = size(lines)
            xt = array([double(lines[i].split()[0]) for i in range(nt)])
            for k in range(nband):
                yt = array([double(lines[i].split()[k+1]) for i in range(nt)])
                nu0 = 1e4*clight/wbd[k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else: # no filter read for existing 'I_MIPS' key
            ttg = tag
            for k in range(nband):
                nu0 = 1e4*clight/wbd[k]
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb    
        # end MIRIM

     elif (inst[ixs] == 'NIRCAM'):
        bnm = ['F070W','F090W','F115W','F140M','F150W','F162M','F164N','F182M','F187N','F200W','F210M', \
               'F212N','F250M','F277W','F300M','F322W2','F323N','F335M','F356W','F360M','F405N','F410M','F430M', \
               'F444W','F460M','F466N','F470N','F480M', ] # band names
        wbd = [0.7,0.9,1.15,1.4,1.5,1.62,1.64,1.82,1.87,2.0,2.1,2.12,2.5,2.77,3.0,3.22,3.23,3.35,3.56,3.6,4.05, \
                4.1,4.3,4.44,4.6,4.66,4.7,4.8] # band center in microns
        nband = size(wbd)
        ntrans = 3000
        dir_band = path + '/NIRCam/JWST_NIRCam.'
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_NIRCAM' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)'
            for k in range(nband):
                fn = dir_band + bnm[k]+'.dat'
                f = open(fn,'r'); lines = f.readlines(); f.close()
                nt = size(lines)
                xt = array([double(lines[i].split()[0]) for i in range(nt)])*1e-4  # convert Ang to microns
                yt = array([double(lines[i].split()[1]) for i in range(nt)])
                nu0 = 1e4*clight/wbd[k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else: # no filter read for existing 'I_NIRCAM' key
            ttg = tag
            for k in range(nband):
                nu0 = 1e4*clight/wbd[k]
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb                    
        # end NIRCam
                
     elif (inst[ixs] == 'NIRCM'):
        bnm = ['F070W','F090W','F115W','F140M','F150W','F150W2','F162M','F164N','F182M','F187N','F200W','F210M', \
               'F212N','F250M','F277W','F300M','F322W2','F323N','F335M','F356W','F360M','F405N','F410M','F430M', \
               'F444W','F460M','F466N','F470N','F480M', ] # band names
        wbd = [0.7,0.9,1.15,1.4,1.5,1.5,1.62,1.64,1.82,1.87,2.0,2.1,2.12,2.5,2.77,3.0,3.22,3.23,3.35,3.56,3.6,4.05, \
                4.1,4.3,4.44,4.6,4.66,4.7,4.8] # band center in microns
        nband = size(wbd)
        ntrans = 3000
        dir_band = path + '/NIRCAM/modAB_mean/filters_only'
        fn = glob.glob(dir_band+'/*_filteronly_ModAB_mean.txt'); nfiles=size(fn) # get filenames
        idb=[]  # order fn list on bnm list
        for j in range(nband): idb.append([i for i,s in enumerate(bnm) if s in fn[j]])  # first get ordering index list
        pz = zip(idb,fn); spz = sorted(pz); fn = [elmt for _, elmt in spz]
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_NIRCAM' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)'
            for k in range(nband):
                f = open(fn[k],'r'); lines = f.readlines(); f.close()
                lines = lines[1:] # to skip comments
                nt = size(lines)
                xt = array([double(lines[i].split()[0]) for i in range(nt)])
                yt = array([double(lines[i].split()[1]) for i in range(nt)])
                nu0 = 1e4*clight/wbd[k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else: # no filter read for existing 'I_NIRCAM' key
            ttg = tag
            for k in range(nband):
                nu0 = 1e4*clight/wbd[k]
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb                    
        # end NIRCM
        
     elif (inst[ixs] == 'PACS'):
        wbd=[70.,100.,160.]     # microns
        bnm = [str(int(wbd[x])) for x in range(3)]
        nband = size(wbd)
        ntrans = 600
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_PACS' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)' 
            dir_band = path + '/PACS'
            fn = dir_band+'/PCalPhotometer_Absorption_FM_v2.fits'
            hh = fits.open(fn)
            if verb == 1: print(hh.info())
            wave = hh[1].data; trans = hh[2].data; hh.close()
            fn = dir_band+'/PCalPhotometer_FilterTransmission_FM_v1.fits'
            hh = fits.open(fn)
            if verb == 1: print(hh.info())
            lamb3 = hh[1].data['wavelength']; t3 =  hh[1].data['transmission']
            lamb1 = hh[2].data['wavelength']; t1 =  hh[2].data['transmission']
            lamb2 = hh[3].data['wavelength']; t2 =  hh[3].data['transmission']
            mx = max([size(lamb1),size(lamb2),size(lamb3)])
            xtt = double([[0.0 for x in range(mx)] for y in range(nband)])
            ytt = double([[0.0 for x in range(mx)] for y in range(nband)])
            tt1 = interp(lamb1,wave,trans)
            tt2 = interp(lamb2,wave,trans)
            tt3 = interp(lamb3,wave,trans)
            ytt[0,:size(lamb1)] = t1*tt1; xtt[0,:size(lamb1)] = lamb1;
            ytt[1,:size(lamb2)] = t2*tt2; xtt[1,:size(lamb2)] = lamb2;
            ytt[2,:size(lamb3)] = t3*tt3; xtt[2,:size(lamb3)] = lamb3;
            for k in range(nband):
                xt = reshape(xtt[k,:],mx); yt = reshape(ytt[k,:],mx)
                yt = yt/xt  # from F Galliano note and PACS doc 
                nu0 = 1e4*clight/wbd[k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else: # no filter read for existing 'I_PACS' key
            ttg = tag
            for k in range(nband):
                nu0 = 1e4*clight/wbd[k]
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                yt = yt/xt  # from F Galliano note and PACS doc 
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb    
        # end PACS
        
     elif (inst[ixs] == 'SPIRE'):
        wbd= [250., 350., 500.]
        bnm = [str(int(wbd[x])) for x in range(3)]
        wbd = double(wbd)
        nband = size(wbd)
        ntrans = 512
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_SPIRE' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = wbd
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)' 
            dir_band = path + '/SPIRE'
            fn = dir_band+'/SPIRE_transmissions.txt'
            f = open(fn,'r'); lines = f.readlines(); f.close()
            if verb == 1: print(fname)
            lines = lines[3:] # to skip comments
            nt = size(lines)
            xt = array([1e4/double(lines[i].split()[0]) for i in range(nt)])  # convert cm-1 to microns
            for k in range(nband):
                yt = double([lines[i].split()[k+1] for i in range(nt)])
                yt = yt*(xt/wbd[k])**2  # from F Galliano note and SPIRE doc (extended source case)
#                yt = yt/xt  # from F Galliano note and SPIRE doc (point source case)
                nu0 = 1e4*clight/wbd[k]
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else: # # no filter read for existing 'I_SPIRE' key
            ttg = tag
            for k in range(nband):
                nu0 = 1e4*clight/wbd[k]
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                yt = yt*(xt/wbd[k])**2  # from F Galliano note and SPIRE doc (extended source case)
#                yt = yt/xt  # from F Galliano note and SPIRE doc (point source case)
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( wbd[k],xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb    
        # end SPIRE
        
     elif (inst[ixs] == 'WMAP'):
        wbd = [ 23., 33., 41., 61., 94. ]   # band center in GHz
        bnm = [str(int(wbd[i]))+'GHz' for i in range(3)]
        wbd = double(wbd)
        nband = size(wbd)
        ntrans = 1000
        itx = where(tag == 'I_'+inst[ixs])[0]; ctg = size(itx)
        if (new_str == 1) or (ctg == 0):  # creating 'I_WMAP' key
            smdat.update( {'I_'+inst[ixs]:str_inst( n1=nband,n2=ntype,ntrans=ntrans )} )
            ttg = array(list(smdat))
            itx = int(where( ttg == 'I_'+inst[ixs])[0])
            smdat[ttg[itx]]['NAME'] = bnm
            smdat[ttg[itx]]['X'] = 1e-5*clight/wbd
            smdat[ttg[itx]]['UNIT'] = 'x(microns) YM(erg/s/cm2/sr) FLX(MJy/sr)' 
            dir_band = path + '/WMAP'
            tmp = glob.glob(dir_band+'/*v3.xcat'); nfiles=size(tmp)
            fn  = [tmp[0],tmp[3],tmp[4],tmp[2],tmp[1]] # sort in frequency order K,Ka,Q,V,W as in wbd
            if verb == 1: print(fn)
            for k in range(nband):
                f = open(fn[k],'r'); lines = f.readlines(); f.close()
                lines = lines[7:] # to skip comments 
                nt = size(lines)
                xt = array([double(lines[i].split()[0]) for i in range(nt)]); xt = 1e-5*clight/xt # convert to microns 
                yt = array([double(lines[i].split()[1]) for i in range(nt)])
                yt = yt/xt  # bolometer case (see F Galliano note) with constant nuInu convention 
                w0 = 1e-5*clight/wbd[k] # microns
                nu0 = 1e9*wbd[k] # Hz
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc( w0,xw,yw[itp,:],xt,yt )
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                ni = size(xi)
                smdat[ttg[itx]]['RR'][k] = rb
                smdat[ttg[itx]]['TRANS']['X'][0:ni,k] = xi
                smdat[ttg[itx]]['TRANS']['Y'][0:ni,k] = yi
        else: # # no filter read for existing 'I_WMAP' key
            ttg = tag
            for k in range(nband):
                w0 = 1e-5*clight/wbd[k] # microns@
                nu0 = 1e9*wbd[k]
                xt = smdat[ttg[itx]]['TRANS']['X'][:,k]
                yt = smdat[ttg[itx]]['TRANS']['Y'][:,k]
                yt = yt/xt  # bolometer case (see F Galliano note) with constant nuInu convention 
                for itp in range(ntype):
                    smdat[ttg[itx]]['YM'][itp,k],y0,tc,rb,xi,yi =  band_cc(wbd[k],xw,yw[itp,:],xt,yt)
                    smdat[ttg[itx]]['FLX'][itp,k] = 1.e17*tc*y0/nu0  # in MJy/sr
                    smdat[ttg[itx]]['CC'][itp,k] = tc
                smdat[ttg[itx]]['RR'][k] = rb   
     # end WMAP

  return smdat  # get_band_flux

